from functools import partial

import sqlalchemy as sa
from sqlalchemy import DECIMAL, func
from sqlalchemy.dialects.postgresql import JSONB, NUMERIC

from sendr_utils import enum_values

from mail.payments.payments.core.entities.arbitrage import ArbitrageStatus
from mail.payments.payments.core.entities.enums import (
    NDS, AcquirerType, ArbitrageVerdict, CommonDataType, FunctionalityType, MerchantOAuthMode, MerchantRole,
    MerchantStatus, ModerationType, OperationKind, OrderKind, OrderSource, PaymentsTestCase, PayStatus, PeriodUnit,
    ProductStatus, RefundStatus, Role, ShopType, TaskState, TaskType, TransactionStatus, WorkerState, WorkerType
)

metadata = sa.MetaData(schema='payments')

SAEnum = partial(sa.Enum, metadata=metadata, values_callable=enum_values)

acquirer_type = SAEnum(AcquirerType, name='acquirer_type')
arbitrage_status = SAEnum(ArbitrageStatus, name='arbitrage_status')
arbitrage_verdict = SAEnum(ArbitrageVerdict, name='arbitrage_verdict')
created_by_source = SAEnum(OrderSource, name='created_by_source')
functionality_type = SAEnum(FunctionalityType, name='functionality_type')
merchant_oauth_mode = SAEnum(MerchantOAuthMode, name='merchant_oauth_mode')
merchant_role = SAEnum(MerchantRole, name='merchant_role')
merchant_status = SAEnum(MerchantStatus, name='merchant_status')
moderation_type = SAEnum(ModerationType, name='moderation_type')
nds = SAEnum(NDS, name='nds')
operation_kind = SAEnum(OperationKind, name='operation_kind')
order_kind = SAEnum(OrderKind, name='order_kind')
pay_by_source = SAEnum(OrderSource, name='pay_by_source')
pay_status = SAEnum(PayStatus, name='pay_order_status')
period_unit = SAEnum(PeriodUnit, name='period_unit')
product_status = SAEnum(ProductStatus, name='product_status')
refund_status = SAEnum(RefundStatus, name='refund_status')
role = SAEnum(Role, name='role')
shop_type = SAEnum(ShopType, name='shop_type')
task_state = SAEnum(TaskState, name='task_state')
task_type = SAEnum(TaskType, name='task_type')
test_case = SAEnum(PaymentsTestCase, name='test_case')
transaction_status = SAEnum(TransactionStatus, name='transaction_status')
worker_state = SAEnum(WorkerState, name='worker_state')
worker_type = SAEnum(WorkerType, name='worker_type')
common_data_type = SAEnum(CommonDataType, name='common_data_type')

DEFAULT_DECIMAL = DECIMAL(12, 2)
DEFAULT_NUMERIC = NUMERIC(precision=12, scale=2)

arbitrages = sa.Table(
    'arbitrages', metadata,
    sa.Column('arbitrage_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), nullable=False),
    sa.Column('order_id', sa.BigInteger(), nullable=False),

    sa.Column('consultation_id', sa.Text(), default=None),
    sa.Column('chat_id', sa.Text(), default=None),
    sa.Column('arbiter_chat_id', sa.Text(), default=None),
    sa.Column('escalate_id', sa.Text(), default=None),
    sa.Column('refund_id', sa.BigInteger(), default=None),
    sa.Column('status', arbitrage_status, default=ArbitrageStatus.CONSULTATION),
    sa.Column('verdict', arbitrage_verdict, default=None),

    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),

    sa.ForeignKeyConstraint(('uid', 'order_id'), ('orders.uid', 'orders.order_id'), ondelete='RESTRICT'),
    sa.ForeignKeyConstraint(('uid', 'refund_id'), ('orders.uid', 'orders.order_id'), ondelete='RESTRICT'),
)

change_log = sa.Table(
    'change_log', metadata,
    sa.Column('uid', sa.BigInteger(), nullable=False),
    sa.Column('revision', sa.BigInteger(), nullable=False),
    sa.Column('operation', operation_kind, nullable=False),
    sa.Column('arguments', JSONB()),
    sa.Column('info', JSONB()),
    sa.Column('changed_at', sa.DateTime(timezone=True), default=func.now()),
)

items = sa.Table(
    'items', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('order_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('product_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('amount', DEFAULT_NUMERIC, nullable=False),
    sa.Column('new_price', DEFAULT_NUMERIC),
    sa.Column('image_id', sa.BigInteger(), nullable=True),
    sa.Column('markup', JSONB(), nullable=True),
)

shops = sa.Table(
    'shops', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('shop_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('name', sa.Text(), nullable=False),
    sa.Column('is_default', sa.Boolean(), nullable=False, default=False),
    sa.Column('shop_type', shop_type, default=ShopType.PROD),
    sa.Column('settings', JSONB()),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),
)

managers = sa.Table(
    'managers', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('domain_login', sa.Text(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False)
)

manager_role = sa.Table(
    'manager_role', metadata,
    sa.Column('manager_uid', sa.BigInteger(), nullable=False),
    sa.Column('role', role),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.PrimaryKeyConstraint('manager_uid', 'role'),
)

merchants = sa.Table(
    'merchants', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('parent_uid', sa.BigInteger()),
    sa.Column('revision', sa.BigInteger(), default=0, nullable=False),
    sa.Column('trustworthy', sa.Boolean(), default=False, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),

    sa.Column('name', sa.Text(), nullable=False),
    sa.Column('api_callback_url', sa.Text()),
    sa.Column('api_callback_params', JSONB(), nullable=False),
    sa.Column('client_id', sa.Text()),
    sa.Column('person_id', sa.Text()),
    sa.Column('contract_id', sa.Text()),
    sa.Column('submerchant_id', sa.Text()),

    sa.Column('status', merchant_status, nullable=False),
    sa.Column('blocked', sa.Boolean(), default=False, nullable=False),

    sa.Column('data', JSONB()),
    sa.Column('data_updated_at', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('data_locked', sa.Boolean(), default=False, nullable=False),

    sa.Column('documents', JSONB()),

    sa.Column('token', sa.Text()),
    sa.Column('merchant_id', sa.Text()),

    sa.Column('acquirer', acquirer_type, nullable=False),

    sa.Column('dialogs_org_id', sa.Text()),
    sa.Column('support_comment', sa.Text()),
    sa.Column('options', JSONB(), nullable=False),
)

merchant_preregistrations = sa.Table(
    'merchant_preregistrations', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('data', JSONB()),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
)

merchant_oauths = sa.Table(
    'merchant_oauths', metadata,
    sa.Column('uid', sa.BigInteger(), nullable=False, primary_key=True),
    sa.Column('shop_id', sa.BigInteger(), nullable=False),  # TODO: PAYBACK-670 (pkey)
    sa.Column('mode', merchant_oauth_mode, nullable=False, primary_key=True),  # TODO: PAYBACK-670 теперь это не pkey

    sa.Column('encrypted_access_token', sa.Text(), nullable=False),
    sa.Column('encrypted_refresh_token', sa.Text(), nullable=False),
    sa.Column('key_version', sa.Integer(), nullable=False),
    sa.Column('expires', sa.DateTime(timezone=True), nullable=False),
    sa.Column('poll', sa.Boolean(), default=True, nullable=False),
    sa.Column('data', JSONB(), nullable=False, default=lambda: {}),

    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),

    sa.PrimaryKeyConstraint('uid', 'mode'),  # TODO: PAYBACK-670 (alter pkey)
    sa.ForeignKeyConstraint(('uid',), ('merchants.uid',), ondelete='CASCADE'),
    sa.ForeignKeyConstraint(('uid', 'shop_id'), ('shops.uid', 'shops.shop_id'), ondelete='CASCADE'),
)

moderations = sa.Table(
    'moderations', metadata,
    sa.Column('moderation_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('moderation_type', moderation_type, nullable=False),
    sa.Column('functionality_type', functionality_type),
    sa.Column('uid', sa.BigInteger(), nullable=False),
    sa.Column('entity_id', sa.BigInteger()),
    sa.Column('revision', sa.BigInteger(), nullable=False),
    sa.Column('approved', sa.Boolean()),
    sa.Column('ignore', sa.Boolean(), nullable=False, default=False),
    sa.Column('unixtime', sa.BigInteger()),
    sa.Column('reason', sa.Text()),
    sa.Column('reasons', JSONB(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
)

orders = sa.Table(
    'orders', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('order_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('shop_id', sa.BigInteger(), nullable=False),
    sa.Column('original_order_id', sa.BigInteger()),
    sa.Column('parent_order_id', sa.BigInteger()),
    sa.Column('revision', sa.BigInteger(), nullable=False),

    sa.Column('kind', order_kind, nullable=False, default=OrderKind.PAY),
    sa.Column('pay_status', pay_status),
    sa.Column('refund_status', refund_status),

    sa.Column('active', sa.Boolean(), nullable=False, default=True),
    sa.Column('autoclear', sa.Boolean(), nullable=False, default=True),
    sa.Column('verified', sa.Boolean(), nullable=False, default=False),
    sa.Column('created_by_source', created_by_source, nullable=False, default=OrderSource.UI),
    sa.Column('pay_by_source', pay_by_source, nullable=False, default=OrderSource.UI),

    sa.Column('closed', sa.DateTime(timezone=True)),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now()),
    sa.Column('held_at', sa.DateTime(timezone=True)),
    sa.Column('pay_status_updated_at', sa.DateTime(timezone=True)),

    sa.Column('caption', sa.Text(), nullable=True),
    sa.Column('description', sa.Text()),
    sa.Column('user_email', sa.Text()),
    sa.Column('user_description', sa.Text()),
    sa.Column('trust_refund_id', sa.Text()),
    sa.Column('customer_uid', sa.BigInteger()),
    sa.Column('return_url', sa.Text()),
    sa.Column('paymethod_id', sa.Text()),

    sa.Column('service_merchant_id', sa.BigInteger()),
    sa.Column('service_client_id', sa.BigInteger()),

    sa.Column('merchant_oauth_mode', merchant_oauth_mode, nullable=True, default=None),
    sa.Column('test', test_case),

    sa.Column('email_message_id', sa.Text(), nullable=True, default=None),
    sa.Column('email_context', JSONB(), nullable=True, default=None),

    sa.Column('customer_subscription_id', sa.BigInteger(), nullable=True, default=None),
    sa.Column('customer_subscription_tx_purchase_token', sa.Text(), nullable=True, default=None),

    sa.Column('data', JSONB(), nullable=False, default=lambda: {}),

    sa.Column('offline_abandon_deadline', sa.DateTime(timezone=True), nullable=True, default=None),

    sa.Column('exclude_stats', sa.Boolean(), nullable=False, default=False),
    sa.Column('acquirer', acquirer_type, default=None),
    sa.Column('commission', sa.Integer()),
)

subscriptions = sa.Table(
    'subscriptions', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('subscription_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('service_merchant_id', sa.BigInteger(), nullable=False),
    sa.Column('service_client_id', sa.BigInteger(), nullable=False),
    sa.Column('product_uuid', sa.Text(), nullable=False),
    sa.Column('title', sa.Text(), nullable=False),
    sa.Column('fiscal_title', sa.Text(), nullable=False),
    sa.Column('nds', nds, nullable=False),
    sa.Column('period_amount', sa.Integer(), nullable=False),
    sa.Column('period_units', period_unit, nullable=False),
    sa.Column('trial_period_amount', sa.Integer(), nullable=True, default=None),
    sa.Column('trial_period_units', period_unit, nullable=True, default=None),
    sa.Column('prices', JSONB(), nullable=False),
    sa.Column('merchant_oauth_mode', merchant_oauth_mode, nullable=True, default=None),

    sa.Column('enabled', sa.Boolean(), nullable=False, default=True),
    sa.Column('revision', sa.BigInteger(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('acquirer', acquirer_type, default=None),
    sa.Column('deleted', sa.Boolean(), nullable=False, default=False),
    sa.Column('data', JSONB(), nullable=False, default=lambda: {}),

    sa.ForeignKeyConstraint(('uid',), ('merchants.uid',), ondelete='RESTRICT'),
    sa.ForeignKeyConstraint(('service_merchant_id',), ('service_merchants.service_merchant_id',), ondelete='RESTRICT'),
    sa.ForeignKeyConstraint(('service_client_id',), ('service_clients.service_client_id',), ondelete='RESTRICT'),
)

customer_subscriptions = sa.Table(
    'customer_subscriptions', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('customer_subscription_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('subscription_id', sa.BigInteger(), nullable=False),
    sa.Column('service_merchant_id', sa.BigInteger(), nullable=False),
    sa.Column('service_client_id', sa.BigInteger(), nullable=False),
    sa.Column('order_id', sa.BigInteger(), nullable=True, default=None),

    sa.Column('user_ip', sa.Text(), nullable=False),
    sa.Column('region_id', sa.BigInteger(), nullable=False),
    sa.Column('quantity', sa.BigInteger(), nullable=False),
    sa.Column('enabled', sa.Boolean(), nullable=False),
    sa.Column('time_until', sa.DateTime(timezone=True), nullable=False),
    sa.Column('time_finish', sa.DateTime(timezone=True), nullable=True, default=None),

    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),

    sa.ForeignKeyConstraint(('uid',), ('merchants.uid',), ondelete='RESTRICT'),
    sa.ForeignKeyConstraint(
        ('uid', 'subscription_id'),
        ('subscriptions.uid', 'subscriptions.subscription_id'),
        ondelete='RESTRICT'
    ),
    sa.ForeignKeyConstraint(('service_merchant_id',), ('service_merchants.service_merchant_id',), ondelete='RESTRICT'),
    sa.ForeignKeyConstraint(('service_client_id',), ('service_clients.service_client_id',), ondelete='RESTRICT'),
)

customer_subscription_transactions = sa.Table(
    'customer_subscription_transactions', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('customer_subscription_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('purchase_token', sa.Text(), primary_key=True, nullable=False),

    sa.Column('payment_status', transaction_status),
    sa.Column('data', JSONB()),

    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),
    sa.Column('trust_order_id', sa.Text(), nullable=True),

    sa.ForeignKeyConstraint(('uid',), ('merchants.uid',), ondelete='RESTRICT'),
    sa.ForeignKeyConstraint(
        ('uid', 'customer_subscription_id',),
        ('customer_subscriptions.uid', 'customer_subscriptions.customer_subscription_id',),
        ondelete='RESTRICT'
    ),
    sa.ForeignKeyConstraint(
        ('uid', 'customer_subscription_id', 'purchase_token'),
        ('orders.uid', 'orders.customer_subscription_id', 'orders.customer_subscription_tx_purchase_token'),
        ondelete='RESTRICT'
    ),
)

products = sa.Table(
    'products', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('product_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('verified', sa.Boolean(), default=False, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now()),
    sa.Column('status', product_status, default=ProductStatus.ACTIVE, nullable=False),
    sa.Column('nds', nds, nullable=False),
    sa.Column('price', DEFAULT_NUMERIC, nullable=False),
    sa.Column('currency', sa.Text(), nullable=False),
    sa.Column('name', sa.Text(), default='', nullable=False),
    sa.Column('revision', sa.BigInteger()),
)

serials = sa.Table(
    'serials', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('next_revision', sa.BigInteger(), nullable=False, default=1),
    sa.Column('next_order_id', sa.BigInteger(), nullable=False, default=1),
    sa.Column('next_tx_id', sa.BigInteger(), nullable=False, default=1),
    sa.Column('next_product_id', sa.BigInteger(), nullable=False, default=1),
    sa.Column('next_subscription_id', sa.BigInteger(), nullable=False, default=1),
    sa.Column('next_customer_subscription_id', sa.BigInteger(), nullable=False, default=1),
    sa.Column('next_shop_id', sa.BigInteger(), nullable=False, default=1),
    sa.Column('next_image_id', sa.BigInteger(), nullable=False, default=1),
)

services = sa.Table(
    'services', metadata,
    sa.Column('service_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('name', sa.Text(), default='', nullable=False),
    sa.Column('slug', sa.Text(), default=None, nullable=True),
    sa.Column('antifraud', sa.Boolean(), default=False, nullable=False),
    sa.Column('order_moderation_enabled', sa.Boolean(), default=True, nullable=False),
    sa.Column('options', JSONB(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('hidden', sa.Boolean(), default=False, nullable=False),
)

service_clients = sa.Table(
    'service_clients', metadata,
    sa.Column('service_client_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('service_id', sa.BigInteger(), nullable=False),
    sa.Column('tvm_id', sa.BigInteger(), unique=True, nullable=False),
    sa.Column('api_callback_url', sa.Text()),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
)

service_merchants = sa.Table(
    'service_merchants', metadata,
    sa.Column('service_merchant_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), nullable=False),
    sa.Column('service_id', sa.BigInteger(), nullable=False),
    sa.Column('enabled', sa.Boolean(), default=False, nullable=False),
    sa.Column('entity_id', sa.Text(), nullable=False),
    sa.Column('description', sa.Text(), nullable=False, default=''),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('deleted', sa.Boolean(), default=False, nullable=False),
    sa.Column('revision', sa.BigInteger()),
)

tasks = sa.Table(
    'tasks', metadata,
    sa.Column('task_id', sa.BigInteger(), primary_key=True, nullable=False),

    sa.Column('task_type', task_type, nullable=False),
    sa.Column('state', task_state, nullable=False),

    sa.Column('action_name', sa.Text()),
    sa.Column('params', JSONB()),
    sa.Column('details', JSONB()),

    sa.Column('retries', sa.Integer(), nullable=False, default=0),

    sa.Column('run_at', sa.DateTime(timezone=True), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),

)

transactions = sa.Table(
    'transactions', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('tx_id', sa.BigInteger(), primary_key=True, nullable=True),
    sa.Column('order_id', sa.BigInteger(), nullable=False),
    sa.Column('revision', sa.BigInteger()),

    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True)),

    sa.Column('status', transaction_status, nullable=False, default=TransactionStatus.ACTIVE),

    sa.Column('trust_purchase_token', sa.Text()),
    sa.Column('trust_payment_url', sa.Text()),
    sa.Column('trust_failed_result', sa.Text()),
    sa.Column('trust_resp_code', sa.Text()),
    sa.Column('trust_payment_id', sa.Text()),
    sa.Column('trust_terminal_id', sa.BigInteger()),

    sa.Column('poll', sa.Boolean(), nullable=False, default=True),
    sa.Column('check_at', sa.DateTime(timezone=True), nullable=False),
    sa.Column('check_tries', sa.Integer(), nullable=False, default=0),
)

workers = sa.Table(
    'workers', metadata,
    sa.Column('worker_id', sa.String(32), nullable=False),
    sa.Column('worker_type', worker_type, nullable=False),
    sa.Column('host', sa.String(), nullable=False),
    sa.Column('state', worker_state, nullable=False),
    sa.Column('heartbeat', sa.DateTime(timezone=True)),
    sa.Column('startup', sa.DateTime(timezone=True)),
    sa.Column('task_id', sa.BigInteger()),
    sa.PrimaryKeyConstraint('worker_id'),
    sa.ForeignKeyConstraint(('task_id',), ('task.id',)),
)

reports = sa.Table(
    'reports', metadata,
    sa.Column('report_id', sa.String(), nullable=False),
    sa.Column('mds_path', sa.String(), nullable=False),
    sa.Column('uid', sa.BigInteger(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('data', JSONB()),
    sa.PrimaryKeyConstraint('report_id'),
)

users = sa.Table(
    'users', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('email', sa.Text()),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),
)

user_roles = sa.Table(
    'user_roles', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('merchant_id', sa.Text(), primary_key=True, nullable=False),
    sa.Column('role', merchant_role, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),
    sa.Column('description', sa.Text()),
    sa.Column('email', sa.Text()),
)

categories = sa.Table(
    'categories', metadata,
    sa.Column('category_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('title', sa.Text(), nullable=False),
    sa.Column('required_acquirer', acquirer_type, nullable=True),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),
)

images = sa.Table(
    'images', metadata,
    sa.Column('uid', sa.BigInteger(), primary_key=True),
    sa.Column('image_id', sa.BigInteger(), primary_key=True),
    sa.Column('url', sa.Text(), nullable=False),
    sa.Column('md5', sa.Text(), nullable=False),
    sa.Column('sha256', sa.Text(), nullable=False),
    sa.Column('stored_path', sa.String(), nullable=True),
    sa.Column('created', sa.DateTime(timezone=True), default=func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=func.now(), nullable=False),
)

common_data = sa.Table(
    'common_data', metadata,
    sa.Column('common_data_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('data_type', common_data_type, nullable=False),
    sa.Column('payload', JSONB(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
)

functionalities = sa.Table(
    'functionalities', metadata,
    sa.Column('uid', sa.BigInteger(), nullable=False),
    sa.Column('functionality_type', functionality_type, nullable=False),
    sa.Column('data', JSONB(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),
    sa.PrimaryKeyConstraint('uid', 'functionality_type'),
)
