import asyncio
from typing import Tuple, Union

from kikimr.public.sdk.python.persqueue.auth import TVMCredentialsProvider
from kikimr.public.sdk.python.persqueue.errors import SessionFailureResult
from kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api import (
    ConsumerConfigurator, PQStreamingAPI, PQStreamingConsumer, PQStreamingProducer, ProducerConfigurator
)

from sendr_qlog import LoggerContext

from mail.payments.payments.conf import settings
from mail.payments.payments.storage.logbroker.exceptions import (
    ClientNotInitializedException, ConsumerStartFailedException, ProducerStartFailedException
)
from mail.payments.payments.utils.runnable import Runnable
from mail.payments.payments.utils.tvm2 import get_tvm2_client


class LogbrokerClient(Runnable):
    """
    Basic wrapper around PQStreamingAPI
    """

    def __init__(self, host: str, port: int, logger: LoggerContext):
        super().__init__()
        self._api = None
        self._host = host
        self._port = port
        self._users: set = set()

    @property
    def api(self) -> PQStreamingAPI:
        if not self._running:
            raise ClientNotInitializedException
        return self._api  # type: ignore

    @property
    def tvm_provider(self) -> TVMCredentialsProvider:
        tvm_client = get_tvm2_client()
        return TVMCredentialsProvider(
            tvm_client=tvm_client,
            destination_alias='logbroker',
            destination_client_id=settings.TVM_LOGBROKER_CLIENT_ID
        )

    def _clear(self):
        self._api = None
        self._users = set()

    async def _run(self):
        self._api = PQStreamingAPI(host=self._host, port=self._port)
        await asyncio.wrap_future(self._api.start())

    async def _close(self):
        for user in self._users:
            await asyncio.wrap_future(user.stop())
        self._api.stop()

    async def create_producer(self,
                              configurator: ProducerConfigurator,
                              ) -> Tuple[PQStreamingProducer, int]:
        producer = self.api.create_producer(configurator, self.tvm_provider)
        start_response = await asyncio.wrap_future(producer.start())
        if isinstance(start_response, SessionFailureResult):
            raise ProducerStartFailedException(start_response.description)
        assert start_response.HasField('init')
        max_seq_no = start_response.init.max_seq_no
        self._users.add(producer)
        return producer, max_seq_no

    async def create_consumer(self, configurator: ConsumerConfigurator) -> PQStreamingConsumer:
        consumer = self.api.create_consumer(configurator, self.tvm_provider)
        start_response = await asyncio.wrap_future(consumer.start())
        if isinstance(start_response, SessionFailureResult):
            raise ConsumerStartFailedException(start_response.description)
        assert start_response.HasField('init')
        self._users.add(consumer)
        return consumer

    async def close_user(self, user: Union[PQStreamingConsumer, PQStreamingProducer]) -> None:
        await asyncio.wrap_future(user.stop())
        self._users.remove(user)
