from typing import Any, Dict, Generic, Iterable, List, TypeVar

import ujson

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.moderation import FastModerationRequest, ModerationResult
from mail.payments.payments.core.exceptions import ModerationUnexpectedUnixtimeTypeError
from mail.payments.payments.storage.logbroker.consumers.base import BaseConsumer
from mail.payments.payments.storage.logbroker.enums import LogbrokerInstallation

T = TypeVar('T')


class BaseModerationConsumer(Generic[T], BaseConsumer[T]):
    def _get_unixtime(self, data: dict) -> int:
        """Ожидает увидеть в результате модерации поле unixtime с целым числом секунд.
        Категорично падает, если пришло что-то плохое, чтобы мы узнали об этом.
        """
        unixtime = data.get('unixtime')
        if isinstance(unixtime, int):
            return unixtime
        if isinstance(unixtime, float):
            self._logger.context_push(moderation_result_raw_data=data)
            self._logger.warn("Expected 'unixtime' to be an 'int' but got 'float'. Will cast to 'int'")
            return int(unixtime)
        raise ModerationUnexpectedUnixtimeTypeError

    def _rows(self, raw_data: bytes) -> Iterable[Dict]:
        """Превратить входную порцию байтов в генератор словарей с ответами модерации."""
        try:
            for row in raw_data.decode('ascii').rstrip().split('\n'):
                yield ujson.loads(row)
        except UnicodeDecodeError:
            self._logger.warning("UnicodeDecodeError in _rows during decode")


class ModerationConsumer(BaseModerationConsumer[ModerationResult]):
    """
    Reads moderation results from topic.
    Moderation requests are provided by ModerationProducer through another topic.

    Data format: MODADVERT-1029
    Example:
    {
        "meta": {
            "client_id": 5678,
            "id": 12345,
            "merchant_id": 543
        },
        "result": {
            "reason": "HighRisk",
            "verdict": "No"
        },
        "service": "pay",
        "type": "merchants"
    }
    """

    INSTALLATION = LogbrokerInstallation(settings.LB_MODERATION_CONSUMER_INSTALLATION)
    CONSUMER = settings.LB_MODERATION_CONSUMER_CONSUMER
    TOPICS = settings.LB_MODERATION_CONSUMER_TOPICS

    def _get_reasons(self, data: Dict[str, Any]) -> List[int]:
        # может стоить использовать библиотеку валидации marshmallow?
        reasons = data.get('reasons', [])
        if isinstance(reasons, list) and all(map(lambda value: isinstance(value, int), reasons)):
            return reasons
        self._logger.context_push(moderation_result_raw_data=data)
        self._logger.error("Wrong 'reasons' value: expected list of integers")
        return []

    @staticmethod
    def _recognize(*, data: Dict[str, Any]) -> bool:
        """В топике могут быть сообщения не для нас. Распознаем, что наше."""
        service_check = data['service'] == 'pay'
        type_check = data['type'] in {'merchants', 'order', 'subscription'}
        env_check = data['meta'].get('env') == settings.MODERATION_ENV if settings.MODERATION_ENV else True
        return service_check and type_check and env_check

    def create_moderation_result(self, data: Dict[str, Any]) -> ModerationResult:
        return ModerationResult(
            moderation_id=data['meta']['id'],
            client_id=data['meta'].get('client_id'),
            submerchant_id=data['meta'].get('merchant_id'),
            approved=data['result']['verdict'] == 'Yes',
            reason=data['result'].get('reason'),
            reasons=self._get_reasons(data['result']),
            unixtime=self._get_unixtime(data),
        )

    def parse_data(self, data: bytes) -> Iterable[ModerationResult]:
        for resp_data in self._rows(data):
            self._logger.context_push(service=resp_data['service'],
                                      type=resp_data['type'],
                                      env=resp_data['meta'].get('env'))

            if not self._recognize(data=resp_data):
                self._logger.info("Skipped moderation result")
                continue
            self._logger.context_push(moderation_result=resp_data)
            self._logger.info("Received moderation result")
            yield self.create_moderation_result(resp_data)


class FastModerationRequestConsumer(BaseModerationConsumer[FastModerationRequest]):
    INSTALLATION = LogbrokerInstallation(settings.LB_MODERATION_CONSUMER_INSTALLATION)
    CONSUMER = settings.LB_FAST_MODERATION_CONSUMER
    TOPICS = (settings.LB_FAST_MODERATION_REQUEST_TOPIC,)

    def parse_data(self, data: bytes) -> Iterable[FastModerationRequest]:
        for req_data in self._rows(data):
            if 'service' not in req_data \
                    or 'type' not in req_data \
                    or 'meta' not in req_data:
                self._logger.info('Message skipped, required fields are missing')
                continue

            self._logger.context_push(service=req_data['service'],
                                      type_=req_data['type'],
                                      meta=req_data['meta'])
            self._logger.info('Received fast moderation request')

            yield FastModerationRequest(
                service=req_data['service'],
                type_=req_data['type'],
                meta=req_data['meta'],
                unixtime=self._get_unixtime(req_data),
            )
