import time
from typing import Dict

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import FunctionalityType, ModerationType
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.moderation import Moderation
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.subscription import Subscription
from mail.payments.payments.storage.logbroker.enums import LogbrokerInstallation
from mail.payments.payments.storage.logbroker.producers.base import BaseProducer
from mail.payments.payments.utils.helpers import without_none


class ModerationProducer(BaseProducer):
    """
    Requests moderation by writing data to topic.
    Moderation results are read by ModerationConsumer from another topic.

    Data format described in: MODADVERT-1029, PAYBACK-51.
    """

    INSTALLATION = LogbrokerInstallation(settings.LB_MODERATION_PRODUCER_INSTALLATION)
    TOPIC = settings.LB_MODERATION_PRODUCER_TOPIC

    @staticmethod
    def _get_offer_type(moderation: Moderation, merchant: Merchant) -> str:
        assert (
            moderation.moderation_type == ModerationType.MERCHANT
            and moderation.functionality_type is not None
        )
        if moderation.functionality_type == FunctionalityType.PAYMENTS:
            assert merchant.acquirer is not None
            offer_settings = merchant.options.offer_settings
            return offer_settings.slug if offer_settings.slug else merchant.acquirer.value
        return moderation.functionality_type.value

    async def write_merchant(self, moderation: Moderation, merchant: Merchant, recheck: bool = False) -> None:
        assert (
            merchant.data is not None
            and merchant.organization is not None
            and merchant.addresses is not None
            and merchant.bank is not None
            and merchant.ceo is not None
            and (merchant.acquirer is not None or moderation.functionality_type == FunctionalityType.YANDEX_PAY)
        )

        offer_type = self._get_offer_type(moderation, merchant)
        meta = {
            'id': moderation.moderation_id,
            'env': settings.MODERATION_ENV or 'default',
        }
        if moderation.functionality_type == FunctionalityType.PAYMENTS:
            # Тинькофф может сообщить модерации о блокировке мерчанта
            meta['merchant_id'] = merchant.get_submerchant_id()
            meta['client_id'] = merchant.client_id
        data = {
            'service': 'pay',
            'type': 'merchants',
            'meta': meta,
            'data': {
                'inn': merchant.organization.inn,
                'type': merchant.organization.type,
                'name': merchant.organization.name,
                'fullName': merchant.organization.full_name,
                'englishName': merchant.organization.english_name,
                'kpp': merchant.organization.kpp,
                'ogrn': merchant.organization.ogrn,
                'offer_type': offer_type,
                'bank': {
                    'account': merchant.bank.account,
                    'bik': merchant.bank.bik,
                },
                'ceo': {
                    'name': merchant.ceo.name,
                    'email': merchant.ceo.email,
                    'phone': merchant.ceo.phone,
                    'surname': merchant.ceo.surname,
                    'patronymic': merchant.ceo.patronymic,
                },
                **{
                    address.type: {
                        'city': address.city,
                        'country': address.country,
                        'home': address.home,
                        'street': address.street,
                        'zip': address.zip,
                    }
                    for address in merchant.addresses
                },
                **{
                    document_type.value: [document.moderation_url for document in documents]
                    for document_type, documents in merchant.get_documents_by_type().items()
                }
            },
            'workflow': 'recheck' if recheck else 'common',
            'unixtime': int(time.time() * 1000),
        }
        return await self.write_dict(data)

    async def write_subscription(self, moderation: Moderation, merchant: Merchant, subscription: Subscription) -> None:
        data = {
            'service': 'pay',
            'type': 'subscription',
            'meta': {
                'id': moderation.moderation_id,
                'client_id': merchant.client_id,
                'subscription_id': subscription.subscription_id,
                'env': settings.MODERATION_ENV or 'default',
            },
            'data': without_none({
                'title': subscription.title,
                'fiscal_title': subscription.fiscal_title,
                'nds': subscription.nds.value,
                'period': subscription.period,
                'trial_period': subscription.trial_period,
                'prices': [
                    {
                        'price': str(price.price),
                        'currency': price.currency,
                        'region_id': price.region_id,
                    }
                    for price in subscription.prices
                ],
            }),
            'unixtime': int(time.time() * 1000),
        }
        return await self.write_dict(data)

    async def write_order(self, moderation: Moderation, merchant: Merchant, order: Order) -> None:
        """Создать сообщение модерации и записать в логброкер."""
        data = self._make_order_moderation_data(moderation=moderation, merchant=merchant, order=order)
        await self.write_dict(data)
        self._logger.context_push(moderation_id=moderation.moderation_id,
                                  order_id=order.order_id,
                                  merchant_uid=merchant.uid,
                                  order_moderation_request=str(data))
        self._logger.info("Send order moderation")

    def _make_order_moderation_data(self, moderation: Moderation, merchant: Merchant, order: Order) -> Dict:
        """Ожидается, что order.items и item.product (для каждого item из order.items) атрибуты установлены."""
        assert order.items is not None
        products = []
        for item in order.items:
            assert item.nds is not None and item.product is not None
            product = {
                'id': item.product_id,
                'name': item.product.name,
                'currency': item.product.currency,
                'price': str(item.price),
                'amount': item.amount,
                'nds': item.nds.value,
                'total_price': item.total_price,
            }
            products.append(product)

        data = {
            'service': 'pay',
            'type': 'order',
            'meta': {
                'id': moderation.moderation_id,
                'client_id': merchant.client_id,
                'order_id': order.order_id,
                'env': settings.MODERATION_ENV or 'default',
            },
            'data': {
                'products': products,
                'total_price': order.price,
            },
            'unixtime': int(time.time() * 1000),
        }
        return data


class FastModerationRequestProducer(ModerationProducer):
    TOPIC = settings.LB_FAST_MODERATION_REQUEST_TOPIC


class FastModerationResponseProducer(BaseProducer):
    INSTALLATION = LogbrokerInstallation(settings.LB_MODERATION_PRODUCER_INSTALLATION)
    TOPIC = settings.LB_FAST_MODERATION_RESPONSE_TOPIC
