import abc
from typing import Iterable, Tuple, TypeVar

from psycopg2.errors import IntegrityError

from sendr_aiopg.query_builder import Filters
from sendr_aiopg.storage import BaseMapper as BMapper

from mail.payments.payments.storage.mappers.mixins import SerialMixin

E = TypeVar('E')


class BaseMapper(SerialMixin, BMapper[E], metaclass=abc.ABCMeta):
    @abc.abstractmethod
    async def create(self, entity: E) -> E:
        raise NotImplementedError

    async def get_or_create(self, entity: E, lookup_fields: Iterable[str], for_update: bool = False) -> Tuple[E, bool]:
        """Get or create Entity

        Assumes mapper implements create method.
        Create method should accept entity instance as single positional argument.

        :param entity: entity to be created if similar entity is not found
        :param lookup_fields: entity field values that are used to find existing entity
        :param for_update: select for update entity if entity is found by select

        :return: created or fetched entity and flag: True if created, False if fetched.

        :raises:
            psycopg2.errors.IntegrityError: lookup does not allow to find existing conflicting entity
        """

        entity_fetched: E

        filters = Filters()
        for field_name in lookup_fields:
            field_value = getattr(entity, field_name)
            filters[field_name] = field_value

        lookup_query, mapper = self._builder.select(filters=filters, for_update=for_update)

        row = await self._query_one(lookup_query, raise_=None)
        if row is not None:
            entity_fetched = mapper(row)
            return entity_fetched, False

        try:
            async with self.conn.begin_nested():
                entity_fetched = await self.create(entity)
                return entity_fetched, True
        except IntegrityError:
            row = await self._query_one(lookup_query, raise_=None)
            if row is None:
                raise
            else:
                entity_fetched = mapper(row)
                return entity_fetched, False
