from datetime import datetime
from typing import AsyncIterable, Iterable, Mapping, Optional

from sqlalchemy.dialects.postgresql import insert

from sendr_aiopg.query_builder import CRUDQueries, Filters

from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import OperationKind
from mail.payments.payments.storage.db.tables import change_log as t_change_log
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.utils.datetime import utcnow
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class ChangeLogDataMapper(SelectableDataMapper):
    entity_class = ChangeLog
    selectable = t_change_log


class ChangeLogDataDumper(TableDataDumper):
    entity_class = ChangeLog
    table = t_change_log


class ChangeLogMapper(BaseMapper):
    name = 'change_log'
    _builder = CRUDQueries(
        t_change_log,
        id_fields=(),
        mapper_cls=ChangeLogDataMapper,
        dumper_cls=ChangeLogDataDumper,
    )

    @staticmethod
    def map(row: Mapping) -> ChangeLog:
        return ChangeLog(
            uid=row['uid'],
            revision=row['revision'],
            operation=row['operation'],
            arguments=row['arguments'],
            info=row['info'],
            changed_at=row['changed_at'],
        )

    @staticmethod
    def unmap(obj: ChangeLog) -> dict:
        return {
            'uid': obj.uid,
            'revision': obj.revision,
            'operation': obj.operation,
            'arguments': obj.arguments,
            'info': obj.info,
        }

    async def create(self, obj: ChangeLog) -> ChangeLog:
        unmapped = self.unmap(obj)
        unmapped['changed_at'] = utcnow()

        query = (
            insert(t_change_log).
            values(**unmapped).
            returning(*t_change_log.c)
        )
        return self.map(await self._query_one(query))

    async def find(self,
                   uid: Optional[int] = None,
                   changed_at_from: Optional[datetime] = None,
                   changed_at_to: Optional[datetime] = None,
                   operations: Optional[Iterable[OperationKind]] = None,
                   offset: Optional[int] = None,
                   limit: Optional[int] = None,
                   ) -> AsyncIterable[ChangeLog]:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_range('changed_at', changed_at_from, changed_at_to)
        filters.add_not_none('operation', operations, lambda column: column.in_(list(operations or [])))
        query, mapper = self._builder.select(
            filters=filters,
            order=('-changed_at',),
            offset=offset,
            limit=limit,
        )
        async for row in self._query(query):
            yield mapper(row)
