from typing import Any, Dict, List

import psycopg2
from sqlalchemy import delete, func

from sendr_aiopg import BaseMapperCRUD
from sendr_aiopg.query_builder import CRUDQueries
from sendr_utils import alist

from mail.payments.payments.core.entities.functionality import FunctionalityData, MerchantFunctionality
from mail.payments.payments.schemas.functionality import FunctionalitySchema
from mail.payments.payments.storage.db.tables import functionalities as t_functionalities
from mail.payments.payments.storage.exceptions import DuplicateFunctionalityStorageError
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class FunctionalityDataMapper(SelectableDataMapper):
    entity_class = MerchantFunctionality
    selectable = t_functionalities

    def map_data(self, data: Dict[str, Any]) -> FunctionalityData:
        return FunctionalitySchema().load(data).data


class FunctionalityDataDumper(TableDataDumper):
    entity_class = MerchantFunctionality
    table = t_functionalities

    def dump_data(self, obj: FunctionalityData) -> Dict[str, Any]:
        return FunctionalitySchema().dump(obj).data


class FunctionalityMapper(BaseMapperCRUD[MerchantFunctionality]):
    name = 'functionality'
    model = MerchantFunctionality

    _builder = CRUDQueries(
        base=t_functionalities,
        id_fields=('uid', 'functionality_type'),
        mapper_cls=FunctionalityDataMapper,
        dumper_cls=FunctionalityDataDumper,
    )

    async def create(self, obj: MerchantFunctionality) -> MerchantFunctionality:
        obj.created = obj.updated = func.now()
        try:
            return await super().create(obj)
        except psycopg2.errors.UniqueViolation:
            raise DuplicateFunctionalityStorageError

    async def save(self, obj: MerchantFunctionality) -> MerchantFunctionality:
        obj.updated = func.now()
        return await super().save(obj, ignore_fields=('uid', 'functionality_type', 'created'))

    async def find_by_uid(self, uid: int) -> List[MerchantFunctionality]:
        return await alist(self.find(filters={'uid': uid}))

    async def delete_by_uid(self, uid: int) -> None:
        query = (
            delete(t_functionalities).
            where(t_functionalities.c.uid == uid)
        )
        await self.conn.execute(query)
