from psycopg2.errors import UniqueViolation
from sqlalchemy import func

from sendr_aiopg.query_builder import CRUDQueries, Filters

from mail.payments.payments.core.entities.image import Image
from mail.payments.payments.storage.db.tables import images as t_images
from mail.payments.payments.storage.exceptions import ImageHashAlreadyExistsStorageError
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class ImageDataMapper(SelectableDataMapper):
    entity_class = Image
    selectable = t_images


class ImageDataDumper(TableDataDumper):
    entity_class = Image
    table = t_images


class ImageMapper(BaseMapper):
    name = 'image'
    _builder = CRUDQueries(
        t_images,
        id_fields=('uid', 'image_id'),
        mapper_cls=ImageDataMapper,
        dumper_cls=ImageDataDumper,
    )

    async def create(self, obj: Image) -> Image:
        async with self.conn.begin():
            obj.image_id = await self._acquire_image_id(obj.uid)
            query, mapper = self._builder.insert(obj, ignore_fields=('created', 'updated'))
            try:
                row = await self._query_one(query)
            except UniqueViolation:
                raise ImageHashAlreadyExistsStorageError
            return mapper(row)

    async def get(self, uid: int, image_id: int, for_update: bool = False) -> Image:
        query, mapper = self._builder.select(id_values=(uid, image_id,), for_update=for_update)
        return mapper(await self._query_one(query, raise_=Image.DoesNotExist))

    async def save(self, obj: Image) -> Image:
        obj.updated = func.now()
        query, mapper = self._builder.update(
            obj,
            ignore_fields=(
                'uid',
                'image_id',
            ),
        )
        async with self.conn.begin_nested():
            try:
                return mapper(await self._query_one(query))
            except UniqueViolation:
                raise ImageHashAlreadyExistsStorageError

    async def get_by_digest(self, uid: int, md5: str, sha256: str, for_update: bool = False) -> Image:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('md5', md5)
        filters.add_not_none('sha256', sha256)
        query, mapper = self._builder.select(filters=filters, for_update=for_update)
        return mapper(await self._query_one(query, raise_=Image.DoesNotExist))

    async def delete(self, obj: Image) -> None:
        query = self._builder.delete(obj)
        await self._query_one(query)
