from typing import AsyncIterable, Mapping, Optional

from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription
from sendr_aiopg.types import ValuesMapping

from mail.payments.payments.core.entities.enums import Role
from mail.payments.payments.core.entities.manager import ManagerRole
from mail.payments.payments.storage.db.tables import manager_role as t_manager_role
from mail.payments.payments.storage.db.tables import managers as t_managers
from mail.payments.payments.storage.exceptions import ManagerRoleNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.storage.mappers.manager import ManagerDataMapper
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class ManagerRoleDataMapper(SelectableDataMapper):
    entity_class = ManagerRole
    selectable = t_manager_role


class ManagerRoleDataDumper(TableDataDumper):
    entity_class = ManagerRole
    table = t_manager_role


class ManagerRoleMapper(BaseMapper):
    name = 'manager_role'

    _related_manager = RelationDescription(
        name='manager',
        base=t_manager_role,
        related=t_managers,
        mapper_cls=ManagerDataMapper,
        base_cols=('manager_uid',),
        related_cols=('uid',),
    )

    _builder = CRUDQueries(
        base=t_manager_role,
        id_fields=('manager_uid',),
        mapper_cls=ManagerRoleDataMapper,
        dumper_cls=ManagerRoleDataDumper,
        related=(_related_manager,),
    )

    @staticmethod
    def _map(row: ValuesMapping, mapper: SelectableDataMapper,
             rel_mappers: Optional[Mapping[str, SelectableDataMapper]] = None) -> ManagerRole:
        role: ManagerRole = mapper(row)
        if rel_mappers is None:
            return role
        role.manager = rel_mappers['manager'](row)
        return role

    async def get(self, manager_uid: int, role: Role) -> ManagerRole:
        query, mapper = self._builder.select(id_values=(manager_uid, role))
        return mapper(await self._query_one(query, raise_=ManagerRoleNotFound))

    async def create(self, manager_role: ManagerRole) -> ManagerRole:
        query, mapper = self._builder.insert(manager_role)
        return mapper(await self._query_one(query))

    async def find(self, manager_uid: Optional[int] = None, with_manager: bool = False) -> AsyncIterable[ManagerRole]:
        filters = Filters()
        filters.add_not_none('manager_uid', manager_uid)
        rel_mappers: Optional[Mapping] = None
        if with_manager:
            query, mapper, rel_mappers = self._builder.select_related(filters=filters)
        else:
            query, mapper = self._builder.select(filters=filters)
        async for row in self._query(query):
            yield self._map(row, mapper, rel_mappers)

    async def delete(self, manager_role: ManagerRole) -> ManagerRole:
        query = self._builder.delete(manager_role)
        return await self._query_one(query)

    async def delete_for_manager(self, manager_uid: int, role: Role) -> None:
        async with self.conn.begin():
            try:
                manager_role = await self.get(manager_uid=manager_uid, role=role)
            except ManagerRoleNotFound:
                return
            await self.delete(manager_role)
