from datetime import datetime
from decimal import Decimal
from operator import eq, gt, lt
from typing import Any, AsyncIterable, Callable, Dict, Iterable, List, Optional, Tuple, cast

from sqlalchemy import Column, alias, and_, asc, case, collate, delete, desc, distinct, func, or_, select
from sqlalchemy.sql import ClauseElement, ColumnElement, FromClause, Selectable
from sqlalchemy.sql.base import Executable

from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription

from mail.payments.payments.core.entities.enums import (
    AcquirerType, MerchantStatus, ModerationStatus, ModerationType, OrderKind, PayStatus, RefundStatus
)
from mail.payments.payments.core.entities.keyset import Keyset, KeysetSortOrder
from mail.payments.payments.core.entities.merchant import Merchant, MerchantAnalyticsStats, MerchantStat
from mail.payments.payments.storage.db.tables import items as t_items
from mail.payments.payments.storage.db.tables import merchants as t_merchants
from mail.payments.payments.storage.db.tables import moderations as t_moderations
from mail.payments.payments.storage.db.tables import orders as t_orders
from mail.payments.payments.storage.db.tables import products as t_products
from mail.payments.payments.storage.db.tables import service_merchants as t_service_merchants
from mail.payments.payments.storage.exceptions import MerchantNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.storage.mappers.merchant.serialization import MerchantDataDumper, MerchantDataMapper
from mail.payments.payments.utils.datetime import utcnow
from mail.payments.payments.utils.db import SelectableDataMapper, create_interval

t_merchants_parent = alias(t_merchants)


class MerchantMapper(BaseMapper):
    name = 'merchant'

    _parent_relation = RelationDescription(
        name='parent',
        base=t_merchants,
        related=t_merchants_parent,
        mapper_cls=MerchantDataMapper,
        base_cols=('parent_uid',),
        related_cols=('uid',),
        outer_join=True,
    )

    _builder = CRUDQueries(
        base=t_merchants,
        id_fields=('uid',),
        mapper_cls=MerchantDataMapper,
        dumper_cls=MerchantDataDumper,
        related=(_parent_relation,),
    )

    @staticmethod
    def _add_parent_filter(query: Selectable,
                           field: str,
                           value: Any,
                           expr: Optional[Callable] = None,
                           ) -> Executable:
        if value is None:
            return query

        def filter_func(f: Column) -> Executable:
            if expr is None:
                return f == value
            return expr(f)

        return query.where(
            or_(
                and_(
                    t_merchants.c.parent_uid == None,  # NOQA
                    filter_func(getattr(t_merchants.c, field))
                ),
                and_(
                    t_merchants.c.parent_uid != None,  # NOQA
                    filter_func(getattr(t_merchants_parent.c, field))
                ),
            )
        )

    def _filter_by_moderation_status(self,
                                     merchants_query: Selectable,
                                     moderation_status: ModerationStatus,
                                     ) -> Executable:
        merchants_query = merchants_query.alias('merchants_query')

        moderations_select_clause = [
            t_moderations.c.moderation_id.label('moderation_id'),
            t_moderations.c.approved.label('approved'),
            t_moderations.c.uid.label('uid'),
        ]

        moderations_subquery = select([func.max(t_moderations.c.moderation_id).label('max_id')]) \
            .select_from(t_moderations) \
            .where(t_moderations.c.moderation_type == ModerationType.MERCHANT) \
            .group_by(t_moderations.c.uid) \
            .alias('moderations_subquery')

        moderations_from_clause = t_moderations \
            .join(moderations_subquery, t_moderations.c.moderation_id == moderations_subquery.c.max_id)

        # select effective moderations
        moderations = select(moderations_select_clause) \
            .select_from(moderations_from_clause) \
            .alias('effective_moderations')

        from_clause = merchants_query \
            .join(moderations, moderations.c.uid == merchants_query.c.Merchant__uid, isouter=True)

        if moderation_status == ModerationStatus.NONE:
            where_clause = from_clause.c.effective_moderations_moderation_id.is_(None)
        elif moderation_status == ModerationStatus.ONGOING:
            where_clause = and_(
                from_clause.c.effective_moderations_approved.is_(None),
                from_clause.c.effective_moderations_moderation_id.isnot(None)
            )
        elif moderation_status == ModerationStatus.APPROVED:
            where_clause = from_clause.c.effective_moderations_approved.is_(True)
        else:
            where_clause = from_clause.c.effective_moderations_approved.is_(False)

        merchants = select(from_clause.columns) \
            .select_from(from_clause) \
            .where(where_clause)
        return merchants

    def _basic_filters(self,
                       uid: Optional[int] = None,
                       name: Optional[str] = None,
                       statuses: Optional[Iterable[MerchantStatus]] = None,
                       acquirer: Optional[AcquirerType] = None,
                       acquirers: Optional[Iterable[AcquirerType]] = None,
                       created_from: Optional[datetime] = None,
                       created_to: Optional[datetime] = None,
                       updated_from: Optional[datetime] = None,
                       updated_to: Optional[datetime] = None) -> Filters:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('status', statuses, lambda field: field.in_(statuses))
        filters.add_not_none(
            'name',
            name,
            lambda f: func.lower(collate(f, "C.UTF-8")).contains(name.lower(), autoescape=True)  # type: ignore
        )
        filters.add_not_none('acquirer', acquirer)
        filters.add_not_none('acquirer', acquirers, lambda field: field.in_(acquirers))

        filters.add_range('created', created_from, created_to)
        filters.add_range('updated', updated_from, updated_to)
        return filters

    def _apply_keyset(self, columns: Dict[str, ColumnElement], keyset: Keyset) -> ClauseElement:
        """
        Given columns C1, C2, C3, C4 and restrictions R1, R2, R3, R4
        Let's build such an OR clause:
        or_(
            selectable.c.C1 > R1,
            and_(selectable.c.C1 == R1, selectable.c.C2 > R2),
            and_(selectable.c.C1 == R1, selectable.c.C2 == R2, selectable.c.C3 > R3),
            and_(selectable.c.C1 == R1, selectable.c.C2 == R2, selectable.c.C3 == R3, selectable.c.C4 > R4),
        )
        NOTE: at least the last column (C4) SHOULD BE unique
        """
        or_clauses = []
        and_clauses: List[ClauseElement] = []
        for column, order, barrier in keyset:
            op = lt if order == 'desc' else gt

            or_clauses.append(
                and_(
                    *and_clauses,
                    op(columns[column], barrier),
                )
            )
            and_clauses.append(eq(columns[column], barrier))

        return or_(*or_clauses)

    def _get_find_order(self,
                        sort_by: Optional[str],
                        descending: Optional[bool],
                        keyset: Optional[Keyset]
                        ) -> List[str]:
        if keyset is None:
            if sort_by is None:
                sort_by = 'uid'
            order = ['-' + sort_by if descending else sort_by]
            if sort_by != 'uid':
                # Для детерминированности добавим в конец сортировку по ключу
                order.append('-uid' if descending else 'uid')
        else:
            assert sort_by is None, 'Should not use sort_by when keyset is set'
            order = []
            for column, sort_order, _ in keyset:
                if sort_order == 'desc':
                    column = '-' + column
                order.append(column)
        return order

    def _prepare_find_query(self,
                            uid: Optional[int] = None,
                            name: Optional[str] = None,
                            username: Optional[str] = None,
                            client_id: Optional[str] = None,
                            submerchant_id: Optional[str] = None,
                            statuses: Optional[Iterable[MerchantStatus]] = None,
                            moderation_status: Optional[ModerationStatus] = None,
                            acquirer: Optional[AcquirerType] = None,
                            acquirers: Optional[Iterable[AcquirerType]] = None,
                            created_from: Optional[datetime] = None,
                            created_to: Optional[datetime] = None,
                            updated_from: Optional[datetime] = None,
                            updated_to: Optional[datetime] = None,
                            limit: Optional[int] = None,
                            offset: Optional[int] = None,
                            sort_by: Optional[str] = None,
                            descending: Optional[bool] = False,
                            keyset: Optional[Keyset] = None,
                            apply_filters_only: bool = False,
                            ) -> Tuple[Executable, SelectableDataMapper]:
        filters = self._basic_filters(
            uid=uid, name=name, statuses=statuses, acquirer=acquirer, acquirers=acquirers, created_from=created_from,
            created_to=created_to, updated_from=updated_from, updated_to=updated_to
        )

        order: List[str] = []
        if apply_filters_only:
            assert limit is None and offset is None and sort_by is None and keyset is None
        else:
            order = self._get_find_order(sort_by=sort_by, descending=descending, keyset=keyset)

        query, mapper, rel_mappers = self._builder.select_related(
            filters=filters,
            offset=offset,
            limit=limit,
            order=order,
        )
        if keyset is not None and not apply_filters_only:
            query = query.where(self._apply_keyset({
                'uid': t_merchants.c.uid,
                'created': t_merchants.c.created,
                'updated': t_merchants.c.updated,
            }, keyset))

        query = self._add_parent_filter(query, 'data', username, lambda f: f['username'].astext == username)
        query = self._add_parent_filter(query, 'client_id', client_id)
        query = self._add_parent_filter(query, 'submerchant_id', submerchant_id)
        if moderation_status:
            query = self._filter_by_moderation_status(query, moderation_status)

        return query, mapper

    def _add_analytics_order(self,
                             columns: Dict[str, ColumnElement],
                             sort_by: Optional[str],
                             descending: Optional[bool],
                             keyset: Optional[Keyset],
                             ) -> List[ColumnElement]:
        func = desc if descending else asc

        if keyset is None:
            if sort_by is None:
                sort_by = 'uid'
            assert sort_by in columns
            order = [func(columns[sort_by])]
            if sort_by != 'uid':
                order.append(func(columns['uid']))
        else:
            assert sort_by is None, 'Should not use sort_by when keyset is set'
            order = []
            for column, sort_order, _ in keyset:
                func = desc if sort_order == 'desc' else asc
                order_clause = func(columns[column])
                order.append(order_clause)

        return order

    async def _prepare_analytics_query(self,
                                       uid: Optional[int] = None,
                                       name: Optional[str] = None,
                                       moderation_status: Optional[ModerationStatus] = None,
                                       acquirer: Optional[AcquirerType] = None,
                                       blocked: Optional[bool] = None,
                                       site_url: Optional[str] = None,
                                       statuses: Optional[Iterable[MerchantStatus]] = None,
                                       created_from: Optional[datetime] = None,
                                       created_to: Optional[datetime] = None,
                                       pay_created_from: Optional[datetime] = None,
                                       pay_created_to: Optional[datetime] = None,
                                       pay_closed_from: Optional[datetime] = None,
                                       pay_closed_to: Optional[datetime] = None,
                                       service_ids: Optional[List[int]] = None,
                                       ) -> Selectable:
        filters = self._basic_filters(
            uid=uid, name=name, acquirer=acquirer, created_from=created_from, created_to=created_to,
            statuses=statuses,
        )
        filters.add_not_none('blocked', blocked)
        filters.add_not_none(
            'data',
            site_url,
            lambda f: func.lower(
                collate(f['organization']['siteUrl'].astext, "C.UTF-8")
            ).contains(cast(str, site_url).lower(), autoescape=True)
        )

        query, mapper, rel_mappers = self._builder.select_related(
            filters=filters,
        )

        if moderation_status is not None:
            query = self._filter_by_moderation_status(query, moderation_status)

        query = query.alias('merchants_query')

        # `_join_orders_stats` выфильтровывает именно заказы созданные нужными сервисами. Этот блок оставляет только
        # тех продавцов, у которых подключен хотя бы один из перечисленных сервисов.
        if service_ids is not None:
            from_clause = (
                query.
                join(
                    t_service_merchants,
                    and_(
                        query.c.Merchant__uid == t_service_merchants.c.uid,
                        t_service_merchants.c.service_id.in_(service_ids),
                        t_service_merchants.c.enabled,
                    ),
                )
            )
            query = (
                select(query.c).
                select_from(from_clause).
                group_by(*query.c).
                alias('merchants_query_filtered_service_id')
            )

        query = self._join_orders_stats(query,
                                        pay_created_from=pay_created_from,
                                        pay_created_to=pay_created_to,
                                        pay_closed_from=pay_closed_from,
                                        pay_closed_to=pay_closed_to,
                                        service_ids=service_ids,
                                        )
        return query, mapper

    def _get_analytics_keyset(self,
                              sort_by: Optional[str],
                              descending: bool,
                              keyset: Optional[Keyset],
                              merchants: List[Tuple[Merchant, MerchantAnalyticsStats]]
                              ) -> Optional[Keyset]:

        def get_attr(item: Tuple[Merchant, MerchantAnalyticsStats], column: str) -> Any:
            tuple_item: Any = item[0]
            if column in ('payments_success', 'money_success'):
                tuple_item = item[1]

            return getattr(tuple_item, column)

        assert sort_by is not None or keyset is not None

        if not merchants:
            return None

        sort_by_items: List[Tuple[str, KeysetSortOrder]] = []
        if keyset is not None:
            for name, order, _ in keyset:
                sort_by_items.append((name, order))
        else:
            order = 'desc' if descending else 'asc'
            assert sort_by is not None
            sort_by_items.append((sort_by, order))
            if sort_by != 'uid':
                sort_by_items.append(('uid', order))

        next_page_keyset: Keyset = []

        prev_column = None
        prev_barrier = None
        for column, order in sort_by_items:
            func = max if order == 'asc' else min

            barrier = func(
                get_attr(item, column)
                for item in merchants
                if prev_column is None or get_attr(item, prev_column) == prev_barrier
            )
            next_page_keyset.append((column, order, barrier))

            prev_column = column
            prev_barrier = barrier

        return next_page_keyset

    def _join_orders_stats(self,
                           query: Selectable,
                           pay_created_from: Optional[datetime],
                           pay_created_to: Optional[datetime],
                           pay_closed_from: Optional[datetime],
                           pay_closed_to: Optional[datetime],
                           service_ids: Optional[List[int]],
                           ) -> Selectable:
        orders_stats = t_orders \
            .join(t_items, and_(t_orders.c.order_id == t_items.c.order_id, t_orders.c.uid == t_items.c.uid)) \
            .join(t_products, and_(t_items.c.product_id == t_products.c.product_id, t_items.c.uid == t_products.c.uid))

        if service_ids:
            orders_stats = orders_stats.join(
                t_service_merchants,
                t_orders.c.service_merchant_id == t_service_merchants.c.service_merchant_id,
                isouter=True,
            )

        orders_stats_group_by_columns = [
            t_orders.c.uid,
            t_orders.c.order_id,
            t_orders.c.kind,
            t_orders.c.pay_status,
            t_orders.c.refund_status,
        ]
        orders_stats_select_columns = [
            *orders_stats_group_by_columns,
            func.sum(t_items.c.amount * t_products.c.price).label('cost'),
        ]
        orders_stats = select(orders_stats_select_columns) \
            .select_from(orders_stats) \
            .group_by(*orders_stats_group_by_columns)

        if pay_created_from is not None:
            orders_stats = orders_stats.where(pay_created_from <= t_orders.c.created)
        if pay_created_to is not None:
            orders_stats = orders_stats.where(t_orders.c.created < pay_created_to)
        if pay_closed_from is not None:
            orders_stats = orders_stats.where(pay_closed_from <= t_orders.c.closed)
        if pay_closed_to is not None:
            orders_stats = orders_stats.where(t_orders.c.closed < pay_closed_to)
        if service_ids:
            orders_stats = orders_stats.where(t_service_merchants.c.service_id.in_(service_ids))
        orders_stats = orders_stats.where(t_orders.c.exclude_stats.is_(False))

        orders_stats = orders_stats.alias('orders_stats')

        analytics_columns = [
            orders_stats.c.uid,
            func.sum(
                case([
                    (
                        orders_stats.c.kind == OrderKind.PAY,
                        1,
                    ),
                ]),
            ).label('payments_total'),
            func.sum(
                case([
                    (
                        and_(
                            orders_stats.c.kind == OrderKind.PAY,
                            orders_stats.c.pay_status == PayStatus.PAID,
                        ),
                        1,
                    ),
                ]),
            ).label('payments_success'),
            func.sum(
                case([
                    (
                        and_(
                            orders_stats.c.kind == OrderKind.REFUND,
                            orders_stats.c.refund_status == RefundStatus.COMPLETED,
                        ),
                        1,
                    ),
                ]),
            ).label('payments_refund'),
        ]

        analytics_columns.extend([
            func.sum(
                case([
                    (
                        and_(
                            orders_stats.c.kind == OrderKind.PAY,
                            orders_stats.c.pay_status == PayStatus.PAID,
                        ),
                        orders_stats.c.cost,
                    ),
                ]),
            ).label('money_success'),
            func.sum(
                case([
                    (
                        and_(
                            orders_stats.c.kind == OrderKind.REFUND,
                            orders_stats.c.refund_status == RefundStatus.COMPLETED,
                        ),
                        orders_stats.c.cost,
                    ),
                ]),
            ).label('money_refund'),
        ])
        analytics_query = select(analytics_columns) \
            .select_from(orders_stats) \
            .group_by(orders_stats.c.uid) \
            .alias('analytics')

        merchants_with_analytics = query.join(analytics_query,
                                              query.c.Merchant__uid == analytics_query.c.uid,
                                              isouter=True)
        return select(
            query.c + (
                func.coalesce(analytics_query.c.payments_total, 0).label('payments_total'),
                func.coalesce(analytics_query.c.payments_success, 0).label('payments_success'),
                func.coalesce(analytics_query.c.payments_refund, 0).label('payments_refund'),
                func.coalesce(analytics_query.c.money_success, 0).label('money_success'),
                func.coalesce(analytics_query.c.money_refund, 0).label('money_refund'),
            )
        ).select_from(merchants_with_analytics)

    async def create(self, obj: Merchant) -> Merchant:
        query, mapper = self._builder.insert(
            obj,
            ignore_fields=(
                'revision',
                'created',
                'updated',
            ),
        )
        return mapper(await self._query_one(query))

    async def find(self,
                   uid: Optional[int] = None,
                   name: Optional[str] = None,
                   username: Optional[str] = None,
                   client_id: Optional[str] = None,
                   submerchant_id: Optional[str] = None,
                   statuses: Optional[Iterable[MerchantStatus]] = None,
                   moderation_status: Optional[ModerationStatus] = None,
                   acquirer: Optional[AcquirerType] = None,
                   acquirers: Optional[Iterable[AcquirerType]] = None,
                   created_from: Optional[datetime] = None,
                   created_to: Optional[datetime] = None,
                   updated_from: Optional[datetime] = None,
                   updated_to: Optional[datetime] = None,
                   limit: Optional[int] = None,
                   offset: Optional[int] = None,
                   sort_by: Optional[str] = None,
                   descending: Optional[bool] = False,
                   keyset: Optional[Keyset] = None,
                   ) -> AsyncIterable[Merchant]:
        """
        Для сортировки нужно использовать или (sort_by, descending), или (keyset).
        Но не одновременно!
        По-умолчанию сортируем по "uid" (asc).
        """
        query, mapper = self._prepare_find_query(uid=uid,
                                                 name=name,
                                                 username=username,
                                                 client_id=client_id,
                                                 submerchant_id=submerchant_id,
                                                 statuses=statuses,
                                                 moderation_status=moderation_status,
                                                 acquirer=acquirer,
                                                 acquirers=acquirers,
                                                 created_from=created_from,
                                                 created_to=created_to,
                                                 updated_from=updated_from,
                                                 updated_to=updated_to,
                                                 limit=limit,
                                                 offset=offset,
                                                 sort_by=sort_by,
                                                 descending=descending,
                                                 keyset=keyset,
                                                 )

        async for row in self._query(query):
            yield mapper(row)

    async def get_found_count(self,
                              uid: Optional[int] = None,
                              name: Optional[str] = None,
                              username: Optional[str] = None,
                              client_id: Optional[str] = None,
                              submerchant_id: Optional[str] = None,
                              statuses: Optional[Iterable[MerchantStatus]] = None,
                              moderation_status: Optional[ModerationStatus] = None,
                              acquirers: Optional[Iterable[AcquirerType]] = None,
                              created_from: Optional[datetime] = None,
                              created_to: Optional[datetime] = None,
                              updated_from: Optional[datetime] = None,
                              updated_to: Optional[datetime] = None,
                              ) -> int:
        query, _ = self._prepare_find_query(uid=uid,
                                            name=name,
                                            username=username,
                                            client_id=client_id,
                                            submerchant_id=submerchant_id,
                                            statuses=statuses,
                                            moderation_status=moderation_status,
                                            acquirers=acquirers,
                                            created_from=created_from,
                                            created_to=created_to,
                                            updated_from=updated_from,
                                            updated_to=updated_to,
                                            apply_filters_only=True,
                                            )

        query = query.alias('merchants_query')

        query = (
            select([func.count()]).
            select_from(query)
        )
        return await self._query_scalar(query)

    async def get_analytics(self,
                            uid: Optional[int] = None,
                            name: Optional[str] = None,
                            moderation_status: Optional[ModerationStatus] = None,
                            acquirer: Optional[AcquirerType] = None,
                            statuses: Optional[Iterable[MerchantStatus]] = None,
                            blocked: Optional[bool] = None,
                            site_url: Optional[str] = None,
                            created_from: Optional[datetime] = None,
                            created_to: Optional[datetime] = None,
                            pay_created_from: Optional[datetime] = None,
                            pay_created_to: Optional[datetime] = None,
                            pay_closed_from: Optional[datetime] = None,
                            pay_closed_to: Optional[datetime] = None,
                            service_ids: Optional[List[int]] = None,
                            limit: Optional[int] = None,
                            sort_by: Optional[str] = None,
                            descending: bool = False,
                            keyset: Optional[Keyset] = None,
                            ) -> Tuple[List[Tuple[Merchant, MerchantAnalyticsStats]], Optional[Keyset]]:
        query, mapper = await self._prepare_analytics_query(
            uid=uid,
            name=name,
            moderation_status=moderation_status,
            acquirer=acquirer,
            statuses=statuses,
            blocked=blocked,
            site_url=site_url,
            created_from=created_from,
            created_to=created_to,
            pay_created_from=pay_created_from,
            pay_created_to=pay_created_to,
            pay_closed_from=pay_closed_from,
            pay_closed_to=pay_closed_to,
            service_ids=service_ids,
        )
        query = query.alias()
        keyset_columns = {
            'uid': query.c.Merchant__uid,
            'created': query.c.Merchant__created,
            'payments_success': query.c.payments_success,
            'money_success': query.c.money_success,
        }
        data_query = select(query.c)
        data_query = data_query.order_by(*self._add_analytics_order(
            columns=keyset_columns,
            sort_by=sort_by,
            descending=descending,
            keyset=keyset,
        ))
        if keyset is not None:
            data_query = data_query.where(self._apply_keyset(columns=keyset_columns, keyset=keyset))

        if limit is not None:
            data_query = data_query.limit(limit)

        merchants = []
        async for row in self._query(data_query):
            merchant = mapper(row)
            stats = MerchantAnalyticsStats(
                payments_total=row['payments_total'],
                payments_success=row['payments_success'],
                payments_refund=row['payments_refund'],
                money_success=row['money_success'],
                money_refund=row['money_refund'],
            )
            merchants.append((merchant, stats))

        if sort_by is not None or keyset is not None:
            keyset = self._get_analytics_keyset(sort_by=sort_by,
                                                descending=descending,
                                                merchants=merchants,
                                                keyset=keyset)
        return merchants, keyset

    async def get_analytics_found(self,
                                  uid: Optional[int] = None,
                                  name: Optional[str] = None,
                                  moderation_status: Optional[ModerationStatus] = None,
                                  statuses: Optional[Iterable[MerchantStatus]] = None,
                                  acquirer: Optional[AcquirerType] = None,
                                  blocked: Optional[bool] = None,
                                  site_url: Optional[str] = None,
                                  created_from: Optional[datetime] = None,
                                  created_to: Optional[datetime] = None,
                                  pay_created_from: Optional[datetime] = None,
                                  pay_created_to: Optional[datetime] = None,
                                  pay_closed_from: Optional[datetime] = None,
                                  pay_closed_to: Optional[datetime] = None,
                                  service_ids: Optional[List[int]] = None,
                                  ) -> int:
        query, _ = await self._prepare_analytics_query(
            uid=uid,
            name=name,
            moderation_status=moderation_status,
            acquirer=acquirer,
            statuses=statuses,
            blocked=blocked,
            site_url=site_url,
            created_from=created_from,
            created_to=created_to,
            pay_created_from=pay_created_from,
            pay_created_to=pay_created_to,
            pay_closed_from=pay_closed_from,
            pay_closed_to=pay_closed_to,
            service_ids=service_ids,
        )
        found = await self._query_scalar(
            select([func.count()]).
            select_from(query.alias('analytics_query'))
        )
        return found

    async def find_by_token(self, token: str, for_update: bool = False) -> Merchant:
        query, mapper = self._builder.select(
            filters={'token': token},
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=MerchantNotFound))

    async def get_by_merchant_id(self, merchant_id: str, for_update: bool = False) -> Merchant:
        query, mapper = self._builder.select(
            filters={'merchant_id': merchant_id},
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=MerchantNotFound))

    async def get(self, uid: int, for_update: bool = False) -> Merchant:
        query, mapper = self._builder.select(
            id_values=(uid,),
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=MerchantNotFound))

    async def get_for_data_update(self, older_than_secs: int, locked_older_than_secs: int) -> Merchant:
        query, mapper = self._builder.select(
            order=[t_merchants.c.data_updated_at.description],
            for_update=True,
            skip_locked=True,
            limit=1
        )

        where_clause = and_(
            t_merchants.c.client_id.isnot(None),
            t_merchants.c.parent_uid.is_(None),
            or_(
                and_(
                    t_merchants.c.data_locked.is_(False),
                    t_merchants.c.data_updated_at < func.now() - create_interval(older_than_secs)
                ),
                and_(
                    t_merchants.c.data_locked.is_(True),
                    t_merchants.c.data_updated_at < func.now() - create_interval(locked_older_than_secs)
                )
            )
        )

        query = query.where(where_clause)
        return mapper(await self._query_one(query, raise_=MerchantNotFound))

    async def get_for_data_update_stats(self) -> Merchant:
        query, mapper = self._builder.select(
            order=[t_merchants.c.data_updated_at.description],
            limit=1
        )
        query = query.where(and_(t_merchants.c.client_id.isnot(None), t_merchants.c.parent_uid.is_(None)))

        return mapper(await self._query_one(query, raise_=MerchantNotFound))

    async def save(self, obj: Merchant) -> Merchant:
        async with self.conn.begin():
            obj.revision = await self._acquire_revision(obj.uid)
            obj.updated = utcnow()
            query, mapper = self._builder.update(obj, ignore_fields=('uid',))
            return mapper(await self._query_one(query))

    def _orders_stats_from_clause(self, isouter: bool = False, with_service: bool = False) -> FromClause:
        result = t_merchants \
            .join(t_orders, t_merchants.c.uid == t_orders.c.uid, isouter=isouter) \
            .join(t_items, and_(t_orders.c.order_id == t_items.c.order_id, t_orders.c.uid == t_items.c.uid),
                  isouter=isouter) \
            .join(t_products, and_(t_items.c.product_id == t_products.c.product_id, t_items.c.uid == t_products.c.uid),
                  isouter=isouter)
        if with_service:
            result = result.join(
                t_service_merchants, t_service_merchants.c.service_merchant_id == t_orders.c.service_merchant_id,
                isouter=True
            )
        return result

    async def batch_orders_stats(self,
                                 default_commission: Decimal,
                                 closed_from: Optional[datetime] = None,
                                 closed_to: Optional[datetime] = None,
                                 service_id: Optional[int] = None,
                                 acquirer: Optional[AcquirerType] = None
                                 ) -> AsyncIterable[MerchantStat]:
        price = func.coalesce(t_items.c.new_price, t_products.c.price)
        commission = case(
            [
                (
                    t_orders.c.kind == OrderKind.PAY,
                    func.coalesce(t_orders.c.commission, int(default_commission * 10000))
                )
            ],
            else_=0,
        )
        select_clause = [
            t_merchants.c.parent_uid.label('parent_uid'),
            t_merchants.c.name.label('name'),
            func.sum(t_items.c.amount * price).label('orders_sum'),
            func.sum(t_items.c.amount * price * commission / 10000).label('commission'),
            t_orders.c.kind.label('orders_kind')
        ]

        where_clause = and_(
            or_(
                and_(
                    t_orders.c.kind == OrderKind.PAY,
                    t_orders.c.pay_status == PayStatus.PAID
                ),
                and_(
                    t_orders.c.kind == OrderKind.REFUND,
                    t_orders.c.refund_status == RefundStatus.COMPLETED
                ),
            ),
            t_orders.c.exclude_stats.is_(False)
        )

        if closed_from is not None:
            where_clause = and_(where_clause, closed_from <= t_orders.c.closed)
        if closed_to is not None:
            where_clause = and_(where_clause, t_orders.c.closed < closed_to)
        if acquirer is not None:
            where_clause = and_(where_clause, t_merchants.c.acquirer == acquirer)
        if service_id is not None:
            where_clause = and_(where_clause, t_service_merchants.c.service_id == service_id)

        query = select(select_clause) \
            .select_from(self._orders_stats_from_clause(with_service=True)) \
            .where(where_clause) \
            .group_by(t_merchants.c.name, t_merchants.c.uid, t_orders.c.kind) \
            .order_by(t_orders.c.kind)

        async for row in self._query(query):
            yield MerchantStat(**row)

    async def orders_stats(self,
                           uid: int,
                           default_commission: Decimal,
                           date_from: Optional[datetime] = None,
                           date_to: Optional[datetime] = None,
                           ) -> MerchantStat:
        where_clause_paid = and_(
            t_orders.c.kind == OrderKind.PAY,
            t_orders.c.pay_status == PayStatus.PAID,
            t_orders.c.uid == uid
        )
        if date_from is not None:
            where_clause_paid = and_(where_clause_paid, date_from <= t_orders.c.closed)
        if date_to is not None:
            where_clause_paid = and_(where_clause_paid, t_orders.c.closed < date_to)

        price = func.coalesce(t_items.c.new_price, t_products.c.price)
        commission = case(
            [
                (
                    t_orders.c.kind == OrderKind.PAY,
                    func.coalesce(t_orders.c.commission, int(default_commission * 10000))
                )
            ],
            else_=0,
        )

        subquery_paid = select([
            t_orders.c.uid,
            func.sum(t_items.c.amount * price).label('orders_sum'),
            func.sum(t_items.c.amount * price * commission / 10000).label('commission'),
            func.count(distinct(t_orders.c.order_id)).label('orders_paid_count'),
        ]).select_from(self._orders_stats_from_clause(isouter=True)) \
            .where(where_clause_paid) \
            .group_by(t_orders.c.uid) \
            .alias('paid')

        where_clause_created = and_(
            t_orders.c.kind == OrderKind.PAY,
            t_orders.c.uid == uid
        )
        if date_from is not None:
            where_clause_created = and_(where_clause_created, date_from <= t_orders.c.created)
        if date_to is not None:
            where_clause_created = and_(where_clause_created, t_orders.c.created < date_to)

        subquery_created = select([
            t_orders.c.uid,
            func.count(t_orders.c.order_id).label('orders_created_count'),
        ]).select_from(t_orders) \
            .where(where_clause_created) \
            .group_by(t_orders.c.uid) \
            .alias('created')

        query = select([
            func.coalesce(subquery_paid.c.orders_sum, 0).label('orders_sum'),
            func.coalesce(subquery_paid.c.commission, 0).label('commission'),
            func.coalesce(subquery_paid.c.orders_paid_count, 0).label('orders_paid_count'),
            func.coalesce(subquery_created.c.orders_created_count, 0).label('orders_created_count'),
        ]).select_from(
            subquery_paid.join(
                subquery_created,
                subquery_paid.c.uid == subquery_created.c.uid,
                full=True
            )
        )
        result = await self._query_one(query)
        return MerchantStat() if not result else MerchantStat(**result)

    async def delete_by_uid(self, uid: int) -> None:
        query = (
            delete(t_merchants).
            where(t_merchants.c.uid == uid)
        )
        await self.conn.execute(query)
