from datetime import date, datetime
from typing import Any, Dict, List, Optional

from sendr_aiopg.types import OptStrList, ValuesMapping
from sendr_utils import enum_value, json_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.document import Document
from mail.payments.payments.core.entities.enums import (
    APICallbackSignMethod, DocumentType, MerchantStatus, MerchantType, OrderSource, PersonType
)
from mail.payments.payments.core.entities.merchant import (
    AddressData, APICallbackParams, BankData, Merchant, MerchantData, MerchantOptions, OrganizationData,
    PaymentSystemsOptions, PersonData
)
from mail.payments.payments.core.entities.service import OfferSettings
from mail.payments.payments.storage.db.tables import merchants as t_merchants
from mail.payments.payments.utils.datetime import date_to_str_optional
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class MerchantDataMapper(SelectableDataMapper):
    entity_class = Merchant
    selectable = t_merchants
    _draft: bool

    @staticmethod
    def _str_to_date_optional(str_: Optional[str]) -> Optional[date]:
        if str_ is None:
            return None
        return date.fromisoformat(str_)

    def map_options(self, data: dict) -> MerchantOptions:
        allowed_order_sources = data.get('allowed_order_sources')
        if allowed_order_sources is not None:
            allowed_order_sources = list(map(OrderSource, allowed_order_sources))

        offer_settings = data.get('offer_settings', {}) or {}

        return MerchantOptions(
            order_offline_abandon_period=data.get(
                'order_offline_abandon_period',
                settings.ORDER_OFFLINE_ABANDON_PERIOD_DEFAULT
            ),
            offer_settings=OfferSettings(
                pdf_template=offer_settings.get('pdf_template'),
                slug=offer_settings.get('slug'),
                data_override=offer_settings.get('data_override') or {},
            ),
            allowed_order_sources=allowed_order_sources,
            allow_create_service_merchants=data.get('allow_create_service_merchants', True),
            hide_commission=data.get('hide_commission', False),
            can_skip_registration=data.get('can_skip_registration', False),
            payment_systems=PaymentSystemsOptions(**data.get('payment_systems', {})),
            order_max_total_price=data.get('order_max_total_price', None)
        )

    def map_data(self, data: dict) -> Optional[MerchantData]:
        if self._draft:
            return None

        merchant_data = MerchantData()

        if 'addresses' in data:
            merchant_data.addresses = [
                AddressData(
                    type=type_,
                    city=address['city'],
                    country=address['country'],
                    home=address.get('home'),
                    street=address['street'],
                    zip=address['zip'],
                )
                for type_, address in data['addresses'].items()
            ]
        if 'bank' in data:
            merchant_data.bank = BankData(
                account=data['bank']['account'],
                bik=data['bank']['bik'],
                correspondent_account=data['bank'].get('correspondentAccount'),
                name=data['bank'].get('name'),
            )

        if 'organization' in data:
            merchant_data.organization = OrganizationData(
                inn=data['organization']['inn'],
                kpp=data['organization']['kpp'],
                name=data['organization']['name'],
                full_name=data['organization']['fullName'],
                english_name=data['organization']['englishName'],
                ogrn=data['organization']['ogrn'],
                schedule_text=data['organization'].get('scheduleText'),
                site_url=data['organization'].get('siteUrl'),
                description=data['organization'].get('description'),
            )
            if data['organization']['type']:
                merchant_data.organization.type = MerchantType(data['organization']['type'])

        if 'persons' in data:
            merchant_data.persons = [
                PersonData(
                    type=PersonType(type_),
                    name=person['name'],
                    email=person['email'],
                    phone=person['phone'],
                    surname=person['surname'],
                    patronymic=person['patronymic'],
                    birth_date=MerchantDataMapper._str_to_date_optional(person.get('birthDate')),
                )
                for type_, person in data['persons'].items()
            ]

        merchant_data.username = data.get('username')
        merchant_data.registered = data.get('registered', True)
        merchant_data.fast_moderation = data.get('fast_moderation', False)

        return merchant_data

    @staticmethod
    def map_documents(documents: Optional[List[dict]]) -> List[Document]:
        if documents is None:
            return []
        return [
            Document(
                document_type=DocumentType(document['document_type']),
                path=document['path'],
                size=document['size'],
                created=datetime.fromisoformat(document['created']),
                name=document.get('name'),
            )
            for document in documents
        ]

    def map_api_callback_params(self, data: dict) -> APICallbackParams:
        return APICallbackParams(
            sign_method=APICallbackSignMethod(data.get('sign_method', APICallbackSignMethod.ASYMMETRIC)),
            secret=data.get('secret')
        )

    def __call__(self, row: ValuesMapping) -> Merchant:
        self._draft = row[self.get_corresponding_column_name('status')] == MerchantStatus.DRAFT
        merchant: Merchant = super().__call__(row)
        if self._draft:
            merchant.draft_data = row[self.get_corresponding_column_name('data')]
        return merchant


class MerchantDataDumper(TableDataDumper):
    entity_class = Merchant
    table = t_merchants

    @staticmethod
    def dump_options(data: MerchantOptions) -> Optional[dict]:
        if data is None:
            return None
        return json_value(data)

    @staticmethod
    def dump_data(data: MerchantData) -> Optional[dict]:
        if data is None:
            return None

        dumped: Dict[str, Any] = {}

        if data.addresses is not None:
            dumped['addresses'] = {
                address.type: {
                    'city': address.city,
                    'country': address.country,
                    'home': address.home,
                    'street': address.street,
                    'zip': address.zip,
                }
                for address in data.addresses
            }

        if data.bank is not None:
            dumped['bank'] = {
                'account': data.bank.account,
                'bik': data.bank.bik,
                'correspondentAccount': data.bank.correspondent_account,
                'name': data.bank.name,
            }

        if data.persons is not None:
            dumped['persons'] = {
                person.type.value: {
                    'name': person.name,
                    'email': person.email,
                    'phone': person.phone,
                    'surname': person.surname,
                    'patronymic': person.patronymic,
                    'birthDate': date_to_str_optional(person.birth_date),
                }
                for person in data.persons
            }

        if data.organization is not None:
            dumped['organization'] = {
                'type': enum_value(data.organization.type),
                'name': data.organization.name,
                'englishName': data.organization.english_name,
                'fullName': data.organization.full_name,
                'inn': data.organization.inn,
                'kpp': data.organization.kpp,
                'ogrn': data.organization.ogrn,
                'scheduleText': data.organization.schedule_text,
                'siteUrl': data.organization.site_url,
                'description': data.organization.description,
            }

        dumped['username'] = data.username
        dumped['registered'] = data.registered
        dumped['fast_moderation'] = data.fast_moderation

        return dumped

    @staticmethod
    def dump_documents(document: List[Document]) -> List[dict]:
        return [
            {
                'document_type': document.document_type.value,
                'path': document.path,
                'size': document.size,
                'created': document.created.isoformat(),
                'name': document.name,
            }
            for document in document
        ]

    def __call__(self, obj: Merchant, skip_fields: OptStrList = None, keep_fields: OptStrList = None) -> ValuesMapping:
        dump_values: dict = dict(super().__call__(obj, skip_fields, keep_fields))
        if not bool(keep_fields) and obj.status == MerchantStatus.DRAFT:
            dump_values['data'] = obj.draft_data
        return dump_values
