from dataclasses import asdict

from sqlalchemy import func

from sendr_aiopg.query_builder import CRUDQueries

from mail.payments.payments.core.entities.merchant_preregistration import (
    MerchantPreregistration, MerchantPreregistrationData, PreregisterData
)
from mail.payments.payments.storage.db.tables import merchant_preregistrations as t_merchant_preregistrations
from mail.payments.payments.storage.exceptions import MerchantPreregistrationNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class MerchantPreregistrationDataMapper(SelectableDataMapper):
    entity_class = MerchantPreregistration
    selectable = t_merchant_preregistrations

    def map_data(self, data: dict) -> MerchantPreregistrationData:
        preregistration_data = MerchantPreregistrationData()
        if data.get('preregister_data') is not None:
            preregistration_data.preregister_data = PreregisterData(
                inn=data['preregister_data']['inn'],
                services=data['preregister_data']['services'],
                categories=data['preregister_data']['categories'],
                require_online=data['preregister_data']['require_online'],
            )
        if data.get('raw_preregister_data') is not None:
            preregistration_data.raw_preregister_data = PreregisterData(
                inn=data['raw_preregister_data']['inn'],
                services=data['raw_preregister_data']['services'],
                categories=data['raw_preregister_data']['categories'],
                require_online=data['raw_preregister_data']['require_online'],
            )
        return preregistration_data


class MerchantPreregistrationDataDumper(TableDataDumper):
    entity_class = MerchantPreregistration
    table = t_merchant_preregistrations

    def dump_data(self, data: MerchantPreregistrationData) -> dict:
        dumped = {}
        if data.preregister_data is not None:
            dumped['preregister_data'] = asdict(data.preregister_data)
        if data.raw_preregister_data is not None:
            dumped['raw_preregister_data'] = asdict(data.raw_preregister_data)
        return dumped


class MerchantPreregistrationMapper(BaseMapper[MerchantPreregistration]):
    name = 'merchant_preregistration'

    _builder = CRUDQueries(
        base=t_merchant_preregistrations,
        id_fields=('uid',),
        mapper_cls=MerchantPreregistrationDataMapper,
        dumper_cls=MerchantPreregistrationDataDumper,
    )

    async def create(self, entity: MerchantPreregistration) -> MerchantPreregistration:
        entity.created = entity.updated = func.now()
        query, mapper = self._builder.insert(entity)
        return mapper(await self._query_one(query))

    async def get(self, uid: int) -> MerchantPreregistration:
        query, mapper = self._builder.select(id_values=(uid,))
        return mapper(await self._query_one(query, raise_=MerchantPreregistrationNotFound))

    async def save(self, entity: MerchantPreregistration) -> MerchantPreregistration:
        entity.updated = func.now()
        query, mapper = self._builder.update(entity, ignore_fields=('created', 'uid'))
        return mapper(await self._query_one(query))
