from datetime import datetime
from typing import AsyncIterable, ClassVar, Optional, Tuple

from sqlalchemy import delete, func, select

from sendr_aiopg.query_builder import CRUDQueries, Filters

from mail.payments.payments.core.entities.functionality import FunctionalityType
from mail.payments.payments.core.entities.moderation import Moderation, ModerationType
from mail.payments.payments.storage.db.tables import moderations as t_moderations
from mail.payments.payments.storage.exceptions import ModerationNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class ModerationDataMapper(SelectableDataMapper):
    entity_class = Moderation
    selectable = t_moderations


class ModerationDataDumper(TableDataDumper):
    entity_class = Moderation
    table = t_moderations


class ModerationMapper(BaseMapper):
    name = 'moderation'
    _builder = CRUDQueries(
        t_moderations,
        id_fields=('moderation_id',),
        mapper_cls=ModerationDataMapper,
        dumper_cls=ModerationDataDumper,
    )

    _create_ignore_fields: ClassVar[Tuple[str, ...]] = ('moderation_id', 'created', 'updated')

    async def count_pending_by_type(self, ignore: Optional[bool] = None) -> AsyncIterable[Tuple[ModerationType, int]]:
        query = (select([t_moderations.c.moderation_type, func.count()]).
                 select_from(t_moderations).
                 where(t_moderations.c.approved.is_(None)).
                 group_by(t_moderations.c.moderation_type))

        if ignore is not None:
            query = query.where(t_moderations.c.ignore == ignore)

        async for row in self._query(query):
            yield row[0], row[1]

    async def create(self, obj: Moderation) -> Moderation:
        query, mapper = self._builder.insert(
            obj,
            ignore_fields=self._create_ignore_fields
        )
        return mapper(await self._query_one(query))

    async def find(self,
                   *,
                   uid: Optional[int] = None,
                   entity_id: Optional[int] = None,
                   moderation_type: Optional[ModerationType] = None,
                   functionality_type: Optional[FunctionalityType] = None,
                   revision: Optional[int] = None,
                   ignore: Optional[bool] = None,
                   has_approved: Optional[bool] = None,
                   for_update: bool = False,
                   ) -> AsyncIterable[Moderation]:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('entity_id', entity_id)
        filters.add_not_none('moderation_type', moderation_type)
        filters.add_not_none('functionality_type', functionality_type)
        filters.add_not_none('revision', revision)
        filters.add_not_none('ignore', ignore)
        if has_approved is not None:
            if has_approved:
                filters['approved'] = lambda field: field != None  # NOQA
            else:
                filters['approved'] = None
        query, mapper = self._builder.select(filters=filters, for_update=for_update)
        async for row in self._query(query):
            yield mapper(row)

    async def get(self, moderation_id: int, raise_: bool = True, for_update: bool = False) -> Moderation:
        query, mapper = self._builder.select(id_values=(moderation_id,), for_update=for_update)
        return mapper(await self._query_one(query, raise_=raise_ and ModerationNotFound))

    async def get_effective(self,
                            uid: int,
                            entity_id: Optional[int] = None,
                            moderation_type: Optional[ModerationType] = ModerationType.MERCHANT,
                            functionality_type: Optional[FunctionalityType] = None,
                            ) -> Optional[Moderation]:
        query, mapper = self._builder.select(
            filters={
                'uid': uid,
                'entity_id': entity_id,
                'moderation_type': moderation_type,
                'functionality_type': functionality_type,
                'approved': lambda field: field != None,  # NOQA
                'ignore': False,
            },
            order=('-moderation_id',),
            limit=1,
        )
        row = await self._query_one(query)
        return None if row is None else mapper(row)

    async def get_for_order(self, uid: int, order_id: int) -> Moderation:
        """
        Получить модерацию заказа, если есть.
        Для заказа в норме должно быть не более одной модерации.
        """
        filters = {'uid': uid, 'entity_id': order_id, 'moderation_type': ModerationType.ORDER, 'ignore': False}
        query, mapper = self._builder.select(filters=filters)
        return mapper(await self._query_one(query=query, raise_=ModerationNotFound))

    async def get_oldest_pending_moderation_created_time(self, moderation_type: ModerationType) -> Optional[datetime]:
        query = (select([t_moderations.c.created]).
                 select_from(t_moderations).
                 where(t_moderations.c.approved.is_(None)).
                 where(t_moderations.c.moderation_type == moderation_type).
                 where(t_moderations.c.ignore.is_(False)).
                 order_by(t_moderations.c.created).
                 limit(1))

        row = await self._query_one(query)
        return row.created if row is not None else None

    async def save(self, obj: Moderation) -> Moderation:
        obj.updated = func.now()
        query, mapper = self._builder.update(
            obj,
            ignore_fields=(
                'moderation_id',
                'uid',
                'moderation_type',
                'entity_id',
                'revision',
                'created',
            )
        )
        return mapper(await self._query_one(query, raise_=ModerationNotFound))

    async def delete_by_uid(self, uid: int) -> None:
        query = (
            delete(t_moderations).
            where(t_moderations.c.uid == uid)
        )
        await self.conn.execute(query)
