from dataclasses import dataclass
from datetime import datetime, timedelta
from decimal import Decimal
from typing import Any, AsyncIterable, Callable, Dict, Iterable, List, Mapping, Optional, Tuple, Union

from sqlalchemy import and_, collate, delete, func, or_, select, tuple_
from sqlalchemy.sql import Selectable, asc, desc
from sqlalchemy.sql.functions import GenericFunction, count

from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription
from sendr_aiopg.types import ValuesMapping

from mail.payments.payments.core.entities.customer_subscription import CustomerSubscription
from mail.payments.payments.core.entities.enums import (
    PAY_METHOD_OFFLINE, PAY_METHODS, PAYMETHOD_ID_OFFLINE, GroupType, OrderKind, OrderSource, PaidOrderStatType,
    PayStatus, RefundStatus
)
from mail.payments.payments.core.entities.merchant import AllPaymentsPoint
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.subscription import Subscription
from mail.payments.payments.storage.db.tables import DEFAULT_DECIMAL
from mail.payments.payments.storage.db.tables import customer_subscriptions as t_customer_subscriptions
from mail.payments.payments.storage.db.tables import items as t_items
from mail.payments.payments.storage.db.tables import orders as t_orders
from mail.payments.payments.storage.db.tables import products as t_products
from mail.payments.payments.storage.db.tables import service_merchants as t_service_merchants
from mail.payments.payments.storage.db.tables import services as t_services
from mail.payments.payments.storage.db.tables import shops as t_shops
from mail.payments.payments.storage.db.tables import subscriptions as t_subscriptions
from mail.payments.payments.storage.exceptions import OrderNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.storage.mappers.item import ItemDataMapper
from mail.payments.payments.storage.mappers.order.serialization import (
    OrderDataDumper, OrderDataMapper, OriginalOrderInfoMapper
)
from mail.payments.payments.storage.mappers.product import ProductDataMapper
from mail.payments.payments.storage.mappers.service.serialization import ServiceDataMapper, ServiceMerchantDataMapper
from mail.payments.payments.storage.mappers.shop import ShopDataMapper
from mail.payments.payments.storage.mappers.subscription.customer_subscription import CustomerSubscriptionDataMapper
from mail.payments.payments.storage.mappers.subscription.subscription import SubscriptionDataMapper
from mail.payments.payments.utils.datetime import utcnow
from mail.payments.payments.utils.db import SelectableDataMapper


class SkipCondition:
    pass


SKIP_CONDITION = SkipCondition()


@dataclass
class FindOrderParams:
    uid: Optional[int] = None
    order_id: Optional[int] = None
    subscription_id: Optional[int] = None
    original_order_id: Optional[int] = None
    service_merchant_id: Optional[int] = None
    created_from: Optional[datetime] = None
    created_to: Optional[datetime] = None
    updated_from: Optional[datetime] = None
    updated_to: Optional[datetime] = None
    held_at_from: Optional[datetime] = None
    held_at_to: Optional[datetime] = None
    price_from: Optional[Decimal] = None
    price_to: Optional[Decimal] = None
    kinds: Optional[Iterable[OrderKind]] = None
    pay_statuses: Optional[Iterable[PayStatus]] = None
    refund_statuses: Optional[Iterable[RefundStatus]] = None
    is_active: Optional[bool] = None
    sort_by: Optional[str] = None
    descending: Optional[bool] = None
    limit: Optional[int] = None
    offset: Optional[int] = None
    text_query: Optional[str] = None
    parent_order_id: Union[Optional[int], SkipCondition] = None
    email_query: Optional[str] = None
    with_customer_subscription: bool = False
    select_customer_subscription: Optional[bool] = False
    exclude_stats: Optional[bool] = None
    pay_method: Optional[str] = None
    created_by_sources: Optional[Iterable[OrderSource]] = None
    service_ids: Optional[Iterable[int]] = None
    customer_subscription_id: Optional[int] = None
    customer_subscription_tx_purchase_token: Optional[str] = None


class OrderMapper(BaseMapper):
    _t_original_orders = t_orders.alias('original_orders')
    name = 'order'

    _shop_relation = RelationDescription(
        name='shop',
        base=t_orders,
        base_cols=('uid', 'shop_id'),
        related=t_shops,
        mapper_cls=ShopDataMapper,
        related_cols=('uid', 'shop_id'),
        outer_join=True,  # means left join to order table
    )
    _item_relation = RelationDescription(
        name='item',
        base=t_orders,
        related=t_items,
        mapper_cls=ItemDataMapper,
        base_cols=('uid', 'order_id'),
        related_cols=('uid', 'order_id'),
        outer_join=True,
    )
    _product_relation = RelationDescription(
        name='product',
        base=t_items,
        related=t_products,
        mapper_cls=ProductDataMapper,
        base_cols=('uid', 'product_id'),
        related_cols=('uid', 'product_id'),
        outer_join=True,
    )
    # https://www.postgresql.org/docs/current/sql-syntax-lexical.html#SQL-SYNTAX-IDENTIFIERS
    # name у relation сокращен, потому что генерируются очень длинные алиасы у колонок
    _customer_subscription_relation = RelationDescription(
        name='cust_sub',
        base=t_orders,
        related=t_customer_subscriptions,
        mapper_cls=CustomerSubscriptionDataMapper,
        base_cols=('uid', 'customer_subscription_id'),
        related_cols=('uid', 'customer_subscription_id'),
        outer_join=True,
    )
    _subscription_relation = RelationDescription(
        name='subscription',
        base=t_customer_subscriptions,
        related=t_subscriptions,
        mapper_cls=SubscriptionDataMapper,
        base_cols=('uid', 'subscription_id'),
        related_cols=('uid', 'subscription_id'),
        outer_join=True,
    )
    _service_merchant_relation = RelationDescription(
        name='service_merchant',
        base=t_orders,
        related=t_service_merchants,
        mapper_cls=ServiceMerchantDataMapper,
        base_cols=('service_merchant_id',),
        related_cols=('service_merchant_id',),
        outer_join=True,
    )
    _service_relation = RelationDescription(
        name='service',
        base=t_service_merchants,
        related=t_services,
        mapper_cls=ServiceDataMapper,
        base_cols=('service_id',),
        related_cols=('service_id',),
        outer_join=True,
    )
    _original_order_relation = RelationDescription(
        name='original_order',
        base=t_orders,
        related=_t_original_orders,
        mapper_cls=OriginalOrderInfoMapper,
        base_cols=('uid', 'original_order_id',),
        related_cols=('uid', 'order_id',),
        outer_join=True,
    )

    _builder_kwargs = dict(
        base=t_orders,
        id_fields=('uid', 'order_id'),
        mapper_cls=OrderDataMapper,
        dumper_cls=OrderDataDumper,
    )

    _builder = CRUDQueries(
        **_builder_kwargs,
        related=(
            _shop_relation,
            _service_merchant_relation,
            _service_relation,
            _original_order_relation,
        ),
    )
    _builder_subscription = CRUDQueries(
        **_builder_kwargs,
        related=(
            _shop_relation,
            _customer_subscription_relation,
            _subscription_relation,
            _service_merchant_relation,
            _service_relation,
            _original_order_relation,
        ),
    )
    _builder_find = CRUDQueries(
        **_builder_kwargs,
        related=(
            _shop_relation,
            _item_relation,
            _product_relation,
            _service_merchant_relation,
            _service_relation,
            _original_order_relation,
        ),
    )
    _builder_find_subscription = CRUDQueries(
        **_builder_kwargs,
        related=(
            _shop_relation,
            _item_relation,
            _product_relation,
            _customer_subscription_relation,
            _subscription_relation,
            _service_merchant_relation,
            _service_relation,
            _original_order_relation,
        ),
    )

    @staticmethod
    def _map_related(row: ValuesMapping,
                     mapper: SelectableDataMapper,
                     rel_mappers: Optional[Mapping[str, SelectableDataMapper]] = None) -> Order:
        order: Order = mapper(row)
        if rel_mappers:
            if order.shop_id is not None and 'shop' in rel_mappers:
                order.shop = rel_mappers['shop'](row)

            if order.service_merchant_id and 'service_merchant' in rel_mappers:
                order.service_merchant = rel_mappers['service_merchant'](row)
                if order.service_merchant is not None:
                    order.service_merchant.service = rel_mappers['service'](row)

            if order.customer_subscription_id is not None and 'cust_sub' in rel_mappers:
                customer_subscription: CustomerSubscription = rel_mappers['cust_sub'](row)
                subscription: Subscription = rel_mappers['subscription'](row)
                order.customer_subscription = customer_subscription
                order.customer_subscription.subscription = subscription

            if order.original_order_id is not None and 'original_order' in rel_mappers:
                order.original_order_info = rel_mappers['original_order'](row)

        return order

    async def count_unfinished_refunds(self) -> AsyncIterable[Tuple[RefundStatus, int]]:
        query = (
            select([t_orders.c.refund_status, func.count()]).
            select_from(t_orders).
            where(
                and_(
                    t_orders.c.kind == OrderKind.REFUND,
                    t_orders.c.pay_status == None,  # NOQA
                    t_orders.c.refund_status.in_((RefundStatus.CREATED, RefundStatus.REQUESTED))
                )
            ).
            group_by(t_orders.c.refund_status)
        )
        async for row in self._query(query):
            yield row[0], row[1]

    async def create(self, obj: Order) -> Order:
        async with self.conn.begin():
            obj.order_id = await self._acquire_order_id(obj.uid)
            obj.revision = await self._acquire_revision(obj.uid)
            obj.created = obj.updated = func.now()
            query, mapper = self._builder.insert(obj)
            order = mapper(await self._query_one(query))
            return order

    def _filter_by_status(self, query: Selectable,
                          pay_statuses: Optional[Iterable[PayStatus]],
                          refund_statuses: Optional[Iterable[RefundStatus]],
                          is_active: Optional[bool]) -> Selectable:
        where_clauses = []
        if pay_statuses:
            clause = and_(
                t_orders.c.pay_status.in_(pay_statuses),
                t_orders.c.active == True  # NOQA
            )
            where_clauses.append(clause)
        if refund_statuses:
            clause = and_(
                t_orders.c.refund_status.in_(refund_statuses),
                t_orders.c.active == True  # NOQA
            )
            where_clauses.append(clause)
        if is_active is not None:
            clause = t_orders.c.active == is_active
            where_clauses.append(clause)
        return query.where(or_(*where_clauses))

    def _filter_by_pay_method(self, query: Selectable, pay_method: Optional[str] = None) -> Selectable:
        if pay_method is not None:
            assert pay_method in PAY_METHODS, f'Invalid pay_method value: {pay_method}'
            query = query.where(t_orders.c.pay_status.in_(PayStatus.ALREADY_PAID_STATUSES))
            if pay_method == PAY_METHOD_OFFLINE:
                query = query.where(t_orders.c.paymethod_id == PAYMETHOD_ID_OFFLINE)
            else:
                query = query.where(
                    or_(
                        t_orders.c.paymethod_id != PAYMETHOD_ID_OFFLINE,
                        t_orders.c.paymethod_id.is_(None)
                    )
                )
        return query

    def _filter_by_customer_subscription(self,
                                         query: Selectable,
                                         customer_subscription_id: Optional[int] = None) -> Selectable:
        if customer_subscription_id is not None:
            query = query.where(t_orders.c.customer_subscription_id == customer_subscription_id)
        return query

    def _filter_by_customer_subscription_tx(self,
                                            query: Selectable,
                                            customer_subscription_id: Optional[int] = None,
                                            customer_subscription_tx_token: Optional[str] = None) -> Selectable:
        if customer_subscription_tx_token is not None:
            # С помощью этого фильтра ищем refund-ы, привязанные к конкретной транзакции
            # ключ транзакции состоит из (uid, customer_subscription_id, purchase_token)
            # поэтому надо убедиться, что при поиске по токену у нас заданы и другие компоненты
            # первичного ключа транзакции.
            # Если customer_subscription_id непустой, то фильтр по нему заполняем в методе
            # _filter_by_customer_subscription()
            assert customer_subscription_id is not None
            query = query.where(t_orders.c.customer_subscription_tx_purchase_token == customer_subscription_tx_token)
        return query

    def _sort_by_price_or_status(self, query: Selectable, sort_by: Optional[str], descending: Optional[bool],
                                 price: GenericFunction) -> Selectable:
        sort_order = desc if descending else asc
        if sort_by == 'price':
            query = query.order_by(sort_order(price))
        if sort_by == 'status':
            query = query.order_by(sort_order(t_orders.c.pay_status), sort_order(t_orders.c.refund_status))
        return query

    def _find_query(
        self,
        filters: Filters,
        sort_by: Optional[str] = None,
        descending: Optional[bool] = None,
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        price_from: Optional[Decimal] = None,
        price_to: Optional[Decimal] = None,
        text_query: Optional[str] = None,
        email_query: Optional[str] = None,
        pay_method: Optional[str] = None,
        pay_statuses: Optional[Iterable[PayStatus]] = None,
        refund_statuses: Optional[Iterable[RefundStatus]] = None,
        is_active: Optional[bool] = None,
        with_customer_subscription: bool = False,
        customer_subscription_id: Optional[int] = None,
        customer_subscription_tx_token: Optional[str] = None,
    ) -> Tuple[Selectable, SelectableDataMapper, Optional[Mapping[str, SelectableDataMapper]]]:
        query: Selectable

        order = None
        if sort_by is not None and sort_by not in ['price', 'status']:
            order = ['-' + sort_by if descending else sort_by]

        price = func.coalesce(t_items.c.new_price, t_products.c.price)
        price = func.sum(t_items.c.amount * price).cast(DEFAULT_DECIMAL).label('price')

        group_by = [
            t_orders.c.uid,
            t_orders.c.order_id,

            t_shops.c.uid,
            t_shops.c.shop_id,

            t_service_merchants.c.uid,
            t_service_merchants.c.service_id,
            t_service_merchants.c.entity_id,
            t_service_merchants.c.description,
            t_service_merchants.c.service_merchant_id,

            t_services.c.name,
            t_services.c.service_id,

            self._t_original_orders.c.uid,
            self._t_original_orders.c.order_id,
        ]

        if with_customer_subscription or customer_subscription_id is not None:
            group_by += [
                t_customer_subscriptions.c.uid,
                t_customer_subscriptions.c.customer_subscription_id,
                t_subscriptions.c.uid,
                t_subscriptions.c.subscription_id
            ]

            query, mapper, rel_mappers = self._builder_find_subscription.select_related(
                filters=filters, order=order, limit=limit, offset=offset,
            )
        else:
            query, mapper, rel_mappers = self._builder_find.select_related(
                filters=filters, order=order, limit=limit, offset=offset
            )

        columns = [*mapper.columns, price]
        if rel_mappers:
            for name, rel_mapper in rel_mappers.items():
                if name not in ('product', 'item'):
                    columns += rel_mapper.columns

        query = (
            query.
            with_only_columns(columns).
            group_by(*group_by)
        )

        # price filters, order by
        if price_from is not None:
            query = query.having(price >= price_from)
        if price_to is not None:
            query = query.having(price < price_to)

        query = self._sort_by_price_or_status(query, sort_by, descending, price)

        if text_query is not None:
            text_query = text_query.lower()
            query = query.where(
                or_(
                    func.lower(collate(t_orders.c.caption, "C.UTF-8")).contains(text_query, autoescape=True),
                    func.lower(collate(t_orders.c.description, "C.UTF-8")).contains(text_query, autoescape=True),
                )
            )

        if email_query is not None:
            email_query = email_query.lower()
            query = query.where(func.lower(t_orders.c.user_email).contains(email_query, autoescape=True))

        query = self._filter_by_status(query, pay_statuses, refund_statuses, is_active)
        query = self._filter_by_pay_method(query, pay_method)
        query = self._filter_by_customer_subscription(query, customer_subscription_id)
        query = self._filter_by_customer_subscription_tx(
            query,
            customer_subscription_id,
            customer_subscription_tx_token
        )

        return query, mapper, rel_mappers

    def _prepare_find_query(self, params: FindOrderParams) -> Tuple[
        Selectable, SelectableDataMapper, Optional[Mapping[str, SelectableDataMapper]]
    ]:
        filters = Filters()

        filters.add_not_none('uid', params.uid)
        filters.add_not_none('order_id', params.order_id)
        filters.add_not_none('original_order_id', params.original_order_id)
        filters.add_not_none('service_merchant_id', params.service_merchant_id)
        filters.add_range('created', params.created_from, params.created_to)
        filters.add_range('updated', params.updated_from, params.updated_to)
        filters.add_range('held_at', params.held_at_from, params.held_at_to)
        filters.add_not_none('kind', params.kinds, lambda field: field.in_(params.kinds))
        filters.add_not_none('exclude_stats', params.exclude_stats, params.exclude_stats)
        filters.add_not_none(
            'customer_subscription_id',
            params.select_customer_subscription,
            lambda field: field.isnot(None) if params.select_customer_subscription else field.is_(None)
        )
        if params.parent_order_id is not SKIP_CONDITION:
            filters['parent_order_id'] = lambda field: (
                field.is_(None)
                if params.parent_order_id is None
                else field == params.parent_order_id
            )
        if params.with_customer_subscription:
            filters.add_not_none('customer_subscription.subscription_id', params.subscription_id)

        filters.add_not_none(
            'created_by_source',
            params.created_by_sources,
            lambda field: field.in_(params.created_by_sources)
        )
        filters.add_not_none(
            'service_merchant.service_id',
            params.service_ids,
            lambda field: field.in_(params.service_ids)
        )

        return self._find_query(
            filters,
            sort_by=params.sort_by,
            descending=params.descending,
            limit=params.limit,
            offset=params.offset,
            price_from=params.price_from,
            price_to=params.price_to,
            text_query=params.text_query,
            email_query=params.email_query,
            pay_statuses=params.pay_statuses,
            refund_statuses=params.refund_statuses,
            pay_method=params.pay_method,
            is_active=params.is_active,
            with_customer_subscription=params.with_customer_subscription,
            customer_subscription_id=params.customer_subscription_id,
            customer_subscription_tx_token=params.customer_subscription_tx_purchase_token,
        )

    async def find(
        self,
        params: Optional[FindOrderParams] = None,
        *,
        iterator: bool = False,
    ) -> AsyncIterable[Order]:
        params = params or FindOrderParams()
        query, mapper, rel_mappers = self._prepare_find_query(params)
        async for row in self._query(query, iterator=iterator):
            yield self._map_related(row, mapper, rel_mappers)

    async def get_found_count(self, params: Optional[FindOrderParams] = None) -> int:
        params = params or FindOrderParams()
        assert all((
            params.limit is None,
            params.offset is None,
            params.sort_by is None,
            params.descending is None
        ))
        query, mapper, rel_mappers = self._prepare_find_query(params)
        query = query.alias('orders_query')
        query = select([func.count()]).select_from(query)
        return await self._query_scalar(query)

    async def get(self,
                  uid: int,
                  order_id: Optional[int] = None,
                  original_order_id: Optional[int] = None,
                  customer_uid: Optional[int] = None,
                  customer_subscription_id: Optional[int] = None,
                  service_merchant_id: Optional[int] = None,
                  kind: Optional[OrderKind] = None,
                  active: Optional[bool] = None,
                  for_update: bool = False,
                  with_customer_subscription: bool = False,
                  select_customer_subscription: Optional[bool] = False,
                  ) -> Order:
        filters = Filters()
        filters.add_not_none('service_merchant_id', service_merchant_id)
        filters.add_not_none('original_order_id', original_order_id)
        filters.add_not_none('customer_uid', customer_uid)
        filters.add_not_none('active', active)
        filters.add_not_none('kind', kind)

        id_values = None
        if uid is not None and order_id is not None:
            id_values = (uid, order_id)
            filters.add_not_none(
                'customer_subscription_id',
                select_customer_subscription,
                lambda field: field.isnot(None) if select_customer_subscription else field.is_(None)
            )
        elif uid is not None and customer_subscription_id is not None:
            filters['uid'] = uid
            filters['customer_subscription_id'] = customer_subscription_id
        else:
            raise RuntimeError('No unique filter condition provided')

        if with_customer_subscription:
            query, mapper, rel_mappers = self._builder_subscription.select_related(
                id_values=id_values, filters=filters, for_update=for_update)
        else:
            query, mapper, rel_mappers = self._builder.select_related(
                id_values=id_values, filters=filters, for_update=for_update)

        row = await self._query_one(query, raise_=OrderNotFound)
        return self._map_related(row, mapper, rel_mappers)

    async def get_unfinished_refund(self, delay: Optional[timedelta] = None) -> Order:
        filters: Dict[str, Any] = {
            'kind': OrderKind.REFUND,
            'pay_status': None,
            'refund_status': RefundStatus.REQUESTED,
        }
        if delay is not None:
            filters['updated'] = lambda field: field < utcnow() - delay
        query, mapper, rel_mappers = self._builder.select_related(
            filters=filters,
            order=('updated',),
            for_update=True,
            skip_locked=True,
            limit=1,
        )
        return self._map_related(await self._query_one(query, raise_=OrderNotFound), mapper, rel_mappers)

    async def get_refunds_for_orders(self,
                                     uid_and_order_id_list: List[Tuple[int, int]],
                                     iterator: bool = False) -> AsyncIterable[Order]:
        query, mapper, rel_mappers = self._prepare_find_query(FindOrderParams(kinds=[OrderKind.REFUND]))
        query = query.where(tuple_(t_orders.c.uid, t_orders.c.original_order_id).in_(uid_and_order_id_list))
        async for row in self._query(query, iterator=iterator):
            yield self._map_related(row, mapper, rel_mappers)

    async def save(self, obj: Order) -> Order:
        async with self.conn.begin():
            obj.revision = await self._acquire_revision(obj.uid, raise_=OrderNotFound)
            obj.updated = utcnow()
            query, mapper = self._builder.update(
                obj,
                ignore_fields=(
                    'uid',
                    'order_id',
                    'kind',
                    'created',
                    'test',
                    'created_by_source'
                ),
            )
            return mapper(await self._query_one(query, raise_=OrderNotFound))

    async def payments_count(self,
                             lower_dt: datetime,
                             upper_dt: datetime,
                             uid: int,
                             group_by: GroupType = GroupType.DAY,
                             pay_status: Optional[PayStatus] = None,
                             order_kind: Optional[OrderKind] = OrderKind.PAY,
                             refund_status: Optional[RefundStatus] = None,
                             pay_method: Optional[str] = None,
                             shop_id: Optional[int] = None,
                             iterator: bool = False,
                             ) -> AsyncIterable[AllPaymentsPoint]:
        x = func.date_trunc(group_by.value, t_orders.c.created).label('x')
        select_clause = [x, func.count().label('y')]

        where_clause = (
            and_(
                lower_dt <= t_orders.c.created,
                t_orders.c.created <= upper_dt,
                t_orders.c.uid == uid,
                t_orders.c.kind == order_kind,
                t_orders.c.exclude_stats.is_(False)
            )
        )

        if pay_status:
            where_clause = and_(where_clause, t_orders.c.pay_status == pay_status)
        if shop_id:
            where_clause = and_(where_clause, t_orders.c.shop_id == shop_id)
        if refund_status:
            where_clause = and_(where_clause, t_orders.c.refund_status == refund_status)
        if shop_id:
            where_clause = and_(where_clause, t_orders.c.shop_id == shop_id)

        query = select(select_clause).select_from(t_orders).where(where_clause)
        query = self._filter_by_pay_method(query, pay_method)
        query = query.group_by(x).order_by(x)

        async for row in self._query(query, iterator=iterator):
            yield AllPaymentsPoint(**row)

    async def _bill_stats(self,
                          lower_dt: datetime,
                          upper_dt: datetime,
                          uid: int,
                          group_by: GroupType = GroupType.DAY,
                          pay_status: Optional[PayStatus] = None,
                          order_kind: Optional[OrderKind] = OrderKind.PAY,
                          refund_status: Optional[RefundStatus] = None,
                          pay_method: Optional[str] = None,
                          shop_id: Optional[int] = None,
                          agg_func: Callable = func.avg,
                          iterator: bool = False,
                          ) -> AsyncIterable[AllPaymentsPoint]:
        x = func.date_trunc(group_by.value, t_orders.c.created).label('x')
        price = func.coalesce(t_items.c.new_price, t_products.c.price)
        select_clause = [
            x,
            func.sum(t_items.c.amount * price).label('y'),
            t_orders.c.order_id.label('order_id')
        ]

        from_clause = t_orders \
            .join(t_items, and_(t_orders.c.order_id == t_items.c.order_id, t_orders.c.uid == t_items.c.uid)) \
            .join(t_products, and_(t_items.c.product_id == t_products.c.product_id, t_items.c.uid == t_products.c.uid))

        where_clause = (
            and_(
                lower_dt <= t_orders.c.created,
                t_orders.c.created <= upper_dt,
                t_orders.c.uid == uid,
                t_orders.c.kind == order_kind,
                t_orders.c.exclude_stats.is_(False)
            )
        )

        if pay_status:
            where_clause = and_(where_clause, t_orders.c.pay_status == pay_status)
        if shop_id:
            where_clause = and_(where_clause, t_orders.c.shop_id == shop_id)
        if refund_status:
            where_clause = and_(where_clause, t_orders.c.refund_status == refund_status)

        subquery = select(select_clause).select_from(from_clause).where(where_clause)
        subquery = self._filter_by_pay_method(subquery, pay_method)
        subquery = subquery.group_by(x, t_orders.c.order_id).alias('subquery')

        x = func.date_trunc(group_by.value, subquery.c.x).label('x')
        y = agg_func(subquery.c.y).label('y')

        select_clause = [x, y]

        query = select(select_clause) \
            .select_from(subquery) \
            .group_by(x) \
            .order_by(x)

        async for row in self._query(query, iterator=iterator):
            yield AllPaymentsPoint(**row)

    def average_bill(self,
                     lower_dt: datetime,
                     upper_dt: datetime,
                     uid: int,
                     group_by: GroupType = GroupType.DAY,
                     pay_status: Optional[PayStatus] = None,
                     order_kind: Optional[OrderKind] = OrderKind.PAY,
                     refund_status: Optional[RefundStatus] = None,
                     pay_method: Optional[str] = None,
                     shop_id: Optional[int] = None,
                     iterator: bool = False,
                     ) -> AsyncIterable[AllPaymentsPoint]:
        return self._bill_stats(
            uid=uid,
            lower_dt=lower_dt,
            upper_dt=upper_dt,
            pay_method=pay_method,
            group_by=group_by,
            pay_status=pay_status,
            shop_id=shop_id,
            order_kind=order_kind,
            refund_status=refund_status,
            agg_func=func.avg,
            iterator=iterator
        )

    def payments_sum(self,
                     lower_dt: datetime,
                     upper_dt: datetime,
                     uid: int,
                     group_by: GroupType = GroupType.DAY,
                     pay_status: Optional[PayStatus] = None,
                     order_kind: Optional[OrderKind] = OrderKind.PAY,
                     refund_status: Optional[RefundStatus] = None,
                     pay_method: Optional[str] = None,
                     shop_id: Optional[int] = None,
                     iterator: bool = False,
                     ) -> AsyncIterable[AllPaymentsPoint]:
        return self._bill_stats(
            uid=uid,
            lower_dt=lower_dt,
            upper_dt=upper_dt,
            pay_method=pay_method,
            group_by=group_by,
            pay_status=pay_status,
            shop_id=shop_id,
            order_kind=order_kind,
            refund_status=refund_status,
            agg_func=func.sum,
            iterator=iterator
        )

    async def paid_count(self, count_type: PaidOrderStatType) -> int:
        where_clause = and_(
            t_orders.c.pay_status == PayStatus.PAID,
            t_orders.c.exclude_stats.is_(False)
        )

        if count_type == PaidOrderStatType.ORDER:
            where_clause = and_(where_clause,
                                t_orders.c.parent_order_id.is_(None),
                                t_orders.c.customer_subscription_id.is_(None))
        elif count_type == PaidOrderStatType.SUBSCRIPTION:
            where_clause = and_(where_clause,
                                t_orders.c.parent_order_id.is_(None),
                                t_orders.c.customer_subscription_id.isnot(None))
        elif count_type == PaidOrderStatType.ORDER_FROM_MULTI_ORDER:
            where_clause = and_(where_clause,
                                t_orders.c.parent_order_id.isnot(None),
                                t_orders.c.customer_subscription_id.is_(None))
        else:
            raise RuntimeError(f'Unknown count_type={count_type.value}')

        query = select([count()]).where(where_clause)
        return int((await self._query_one(query))[0])

    async def get_abandoned_order(self, for_update: bool = False) -> Order:
        filters = Filters()
        filters['pay_status'] = PayStatus.NEW
        filters['offline_abandon_deadline'] = lambda field: field < func.now()

        query, mapper, rel_mappers = self._builder.select_related(
            filters=filters,
            order=('offline_abandon_deadline',),
            for_update=for_update,
            limit=1,
        )
        return self._map_related(await self._query_one(query, raise_=OrderNotFound), mapper, rel_mappers)

    async def get_oldest_non_terminal_pay_status_updated(
        self,
        uid_black_list: List[int] = []
    ) -> AsyncIterable[Tuple[PayStatus, datetime]]:
        query = (
            select([t_orders.c.pay_status, func.min(t_orders.c.pay_status_updated_at)]).
            select_from(t_orders).
            where(
                and_(
                    t_orders.c.pay_status.in_(PayStatus.NON_TERMINAL_STATUSES),
                    t_orders.c.uid.notin_(uid_black_list),
                )
            ).
            group_by(t_orders.c.pay_status)
        )
        async for row in self._query(query):
            yield row[0], row[1]

    async def delete_by_uid(self, uid: int) -> None:
        query = (
            delete(t_orders).
            where(t_orders.c.uid == uid)
        )
        await self.conn.execute(query)
