from typing import Mapping

from sqlalchemy.dialects.postgresql import insert

from sendr_aiopg.query_builder import CRUDQueries

from mail.payments.payments.core.entities.serial import Serial
from mail.payments.payments.storage.db.tables import serials as t_serials
from mail.payments.payments.storage.exceptions import SerialNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class SerialDataMapper(SelectableDataMapper):
    entity_class = Serial
    selectable = t_serials


class SerialDataDumper(TableDataDumper):
    entity_class = Serial
    table = t_serials


class SerialMapper(BaseMapper):
    name = 'serial'
    _builder = CRUDQueries(
        t_serials,
        id_fields=('uid',),
        mapper_cls=SerialDataMapper,
        dumper_cls=SerialDataDumper,
    )

    @staticmethod
    def map(row: Mapping) -> Serial:
        return Serial(
            uid=row['uid'],
            next_revision=row['next_revision'],
            next_order_id=row['next_order_id'],
            next_tx_id=row['next_tx_id'],
            next_product_id=row['next_product_id'],
        )

    @staticmethod
    def unmap(obj: Serial) -> dict:
        return {
            'uid': obj.uid,
        }

    async def get(self, uid: int, for_update: bool = False) -> Serial:
        query, mapper = self._builder.select(id_values=(uid,), for_update=for_update)
        return mapper(await self._query_one(query, raise_=SerialNotFound))

    async def create(self, obj: Serial) -> Serial:
        query = (
            insert(t_serials).
            values(**self.unmap(obj)).
            returning(*t_serials.c)
        )
        return self.map(await self._query_one(query))
