from typing import Optional

from sendr_utils import json_value

from mail.payments.payments.core.entities.enums import AcquirerType, OrderSource
from mail.payments.payments.core.entities.service import (
    OfferSettings, Service, ServiceClient, ServiceMerchant, ServiceOptions
)
from mail.payments.payments.storage.db.tables import service_clients as t_service_clients
from mail.payments.payments.storage.db.tables import service_merchants as t_service_merchants
from mail.payments.payments.storage.db.tables import services as t_services
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class ServiceDataMapper(SelectableDataMapper):
    entity_class = Service
    selectable = t_services

    def map_options(self, data: dict) -> ServiceOptions:
        required_acquirer = data.get('required_acquirer')
        if required_acquirer is not None:
            required_acquirer = AcquirerType(required_acquirer)

        allowed_order_sources = data.get('allowed_order_sources')
        if allowed_order_sources is not None:
            allowed_order_sources = list(map(OrderSource, allowed_order_sources))

        offer_settings = data.get('offer_settings', {}) or {}

        return ServiceOptions(
            service_fee=data.get('service_fee'),
            offer_settings=OfferSettings(
                pdf_template=offer_settings.get('pdf_template'),
                slug=offer_settings.get('slug'),
                data_override=offer_settings.get('data_override') or {},
            ),
            allow_create_service_merchants=data.get('allow_create_service_merchants', True),
            hide_commission=data.get('hide_commission', False),
            allow_payment_mode_recurrent=data.get('allow_payment_mode_recurrent', False),
            allow_payment_without_3ds=data.get('allow_payment_without_3ds', False),
            allowed_order_sources=allowed_order_sources,
            required_acquirer=required_acquirer,
            require_online=data.get('require_online', False),
            can_skip_registration=data.get('can_skip_registration', False),
            icon_url=data.get('icon_url', None),
            commission=data.get('commission'),
        )


class ServiceDataDumper(TableDataDumper):
    entity_class = Service
    table = t_services

    @staticmethod
    def dump_options(data: ServiceOptions) -> Optional[dict]:
        if data is None:
            return None
        return json_value(data)


class ServiceClientDataMapper(SelectableDataMapper):
    entity_class = ServiceClient
    selectable = t_service_clients


class ServiceClientDataDumper(TableDataDumper):
    entity_class = ServiceClient
    table = t_service_clients


class ServiceMerchantDataMapper(SelectableDataMapper):
    entity_class = ServiceMerchant
    selectable = t_service_merchants


class ServiceMerchantDataDumper(TableDataDumper):
    entity_class = ServiceMerchant
    table = t_service_merchants
