from typing import AsyncIterable, Mapping, Optional

from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription
from sendr_aiopg.types import ValuesMapping

from mail.payments.payments.core.entities.service import ServiceClient
from mail.payments.payments.storage.db.tables import service_clients as t_service_clients
from mail.payments.payments.storage.db.tables import services as t_services
from mail.payments.payments.storage.exceptions import ServiceClientNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.storage.mappers.service.serialization import (
    ServiceClientDataDumper, ServiceClientDataMapper, ServiceDataMapper
)
from mail.payments.payments.utils.db import SelectableDataMapper


class ServiceClientMapper(BaseMapper):
    name = 'service_client'
    _service_relation = RelationDescription(
        name='service',
        base=t_service_clients,
        related=t_services,
        mapper_cls=ServiceDataMapper,
        base_cols=('service_id',),
        related_cols=('service_id',),
    )
    _builder = CRUDQueries(
        base=t_service_clients,
        id_fields=('service_client_id',),
        mapper_cls=ServiceClientDataMapper,
        dumper_cls=ServiceClientDataDumper,
        related=(_service_relation,)
    )

    @staticmethod
    def map(row: ValuesMapping,
            mapper: SelectableDataMapper,
            rel_mappers: Optional[Mapping[str, SelectableDataMapper]] = None) -> ServiceClient:
        service_client: ServiceClient = mapper(row)
        if rel_mappers:
            service_client.service = rel_mappers['service'](row)
        return service_client

    async def create(self, obj: ServiceClient) -> ServiceClient:
        query, mapper = self._builder.insert(
            obj,
            ignore_fields=(
                'service_client_id',
                'created',
                'updated',
            ),
        )
        return mapper(await self._query_one(query))

    async def find(self, service_id: Optional[int] = None, with_service: bool = False) -> AsyncIterable[ServiceClient]:
        filters = Filters()
        filters.add_not_none('service_id', service_id)

        rel_mappers: Optional[Mapping] = None
        if with_service:
            query, mapper, rel_mappers = self._builder.select_related(filters=filters)
        else:
            query, mapper = self._builder.select(filters=filters)

        async for row in self._query(query):
            yield self.map(row, mapper, rel_mappers)

    async def get(self, service_client_id: int) -> ServiceClient:
        query, mapper = self._builder.select(id_values=(service_client_id,))
        return mapper(await self._query_one(query, raise_=ServiceClientNotFound))
