from typing import AsyncIterable, Mapping, Optional

from sqlalchemy import delete

from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription
from sendr_aiopg.types import ValuesMapping

from mail.payments.payments.core.entities.service import ServiceMerchant
from mail.payments.payments.storage.db.tables import service_merchants as t_service_merchants
from mail.payments.payments.storage.db.tables import services as t_service
from mail.payments.payments.storage.exceptions import ServiceMerchantNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.storage.mappers.service.serialization import (
    ServiceDataMapper, ServiceMerchantDataDumper, ServiceMerchantDataMapper
)
from mail.payments.payments.utils.datetime import utcnow
from mail.payments.payments.utils.db import SelectableDataMapper


class ServiceMerchantMapper(BaseMapper):
    name = 'service_merchant'
    _service_relation = RelationDescription(
        name='service',
        base=t_service_merchants,
        related=t_service,
        mapper_cls=ServiceDataMapper,
        base_cols=('service_id',),
        related_cols=('service_id',),
    )
    _builder = CRUDQueries(
        base=t_service_merchants,
        id_fields=('service_merchant_id',),
        mapper_cls=ServiceMerchantDataMapper,
        dumper_cls=ServiceMerchantDataDumper,
        related=(_service_relation,),
    )

    @staticmethod
    def map(row: ValuesMapping, mapper: SelectableDataMapper,
            rel_mappers: Optional[Mapping[str, SelectableDataMapper]] = None) -> ServiceMerchant:
        service_merchant: ServiceMerchant = mapper(row)
        if rel_mappers:
            service_merchant.service = rel_mappers['service'](row)
        return service_merchant

    async def create(self, obj: ServiceMerchant) -> ServiceMerchant:
        async with self.conn.begin():
            obj.revision = await self._acquire_revision(obj.uid)
            query, mapper = self._builder.insert(
                obj,
                ignore_fields=(
                    'service_merchant_id',
                    'created',
                    'updated',
                    'deleted',
                ),
            )
            return mapper(await self._query_one(query))

    async def find(self,
                   uid: Optional[int] = None,
                   service_id: Optional[int] = None,
                   entity_id: Optional[str] = None,
                   ignore_deleted: bool = True,
                   with_service: bool = False,
                   limit: Optional[int] = None,
                   iterator: bool = False
                   ) -> AsyncIterable[ServiceMerchant]:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('service_id', service_id)
        filters.add_not_none('entity_id', entity_id)
        if ignore_deleted:
            filters.add_not_none('deleted', False)

        rel_mappers: Optional[Mapping[str, SelectableDataMapper]] = None
        if with_service:
            query, mapper, rel_mappers = self._builder.select_related(filters=filters, limit=limit)
        else:
            query, mapper = self._builder.select(filters=filters, limit=limit)

        async for row in self._query(query, iterator=iterator):
            yield self.map(row, mapper, rel_mappers)

    async def get(self,
                  service_merchant_id: Optional[int] = None,
                  service_id: Optional[int] = None,
                  uid: Optional[int] = None,
                  entity_id: Optional[str] = None,
                  fetch_service: bool = True,
                  for_update: bool = False,
                  skip_locked: bool = False,
                  ignore_deleted: bool = True) -> ServiceMerchant:
        """
        Get ServiceMerchant entity by service_merchant_id or (uid, service_id, entity_id).
        ignore_deleted flag will exclude entities with deleted flag
        """
        if service_merchant_id is None and any((uid is None, service_id is None, entity_id is None)):
            raise RuntimeError('No unique filter condition provided')

        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('service_id', service_id)
        filters.add_not_none('entity_id', entity_id)
        if ignore_deleted:
            filters.update({'deleted': False})

        rel_mappers: Optional[Mapping] = None

        kwargs = {
            'filters': filters,
            'for_update': for_update,
            'skip_locked': skip_locked
        }
        if service_merchant_id is not None:
            kwargs['id_values'] = (service_merchant_id,)

        if fetch_service:
            query, mapper, rel_mappers = self._builder.select_related(**kwargs)
        else:
            query, mapper = self._builder.select(**kwargs)

        row = await self._query_one(query, raise_=ServiceMerchantNotFound)

        service_merchant: ServiceMerchant = mapper(row)
        if rel_mappers:
            service_merchant.service = rel_mappers['service'](row)

        return service_merchant

    async def save(self, obj: ServiceMerchant) -> ServiceMerchant:
        async with self.conn.begin():
            obj.updated = utcnow()
            obj.revision = await self._acquire_revision(obj.uid, raise_=ServiceMerchantNotFound)
            query, mapper = self._builder.update(
                obj,
                ignore_fields=(
                    'service_merchant_id',
                    'created',
                    'deleted',
                ),
            )
            return mapper(await self._query_one(query))

    async def delete(self, obj: ServiceMerchant) -> ServiceMerchant:
        """Мы не хотим по-настоящему удалять записи - мы помечаем их флагом deleted и учитываем этот флаг потом."""
        obj.deleted = True
        obj.updated = utcnow()
        obj.revision = await self._acquire_revision(obj.uid, raise_=ServiceMerchantNotFound)
        query, mapper = self._builder.update(obj, only_fields=('deleted', 'updated', 'revision'))
        return mapper(await self._query_one(query))

    async def delete_by_uid(self, uid: int) -> None:
        query = (
            delete(t_service_merchants).
            where(t_service_merchants.c.uid == uid)
        )
        await self.conn.execute(query)
