from typing import AsyncIterable, Mapping, Optional

from psycopg2.errors import IntegrityError  # NOQA
from sqlalchemy import func

from sendr_aiopg.query_builder import CRUDQueries, Filters, OptStrList, OptValuesList, RelationDescription
from sendr_aiopg.types import ValuesMapping

from mail.payments.payments.core.entities.enums import ShopType
from mail.payments.payments.core.entities.shop import Shop, ShopSettings
from mail.payments.payments.storage.db.tables import merchant_oauths as t_merchant_oauths
from mail.payments.payments.storage.db.tables import shops as t_shops
from mail.payments.payments.storage.exceptions import ShopNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.storage.mappers.merchant_oauth import MerchantOAuthDataMapper
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper

__all__ = (
    'ShopMapper',
    'ShopDataDumper',
    'ShopDataMapper',
)


class ShopDataMapper(SelectableDataMapper):
    entity_class = Shop
    selectable = t_shops

    @staticmethod
    def map_settings(data: dict) -> ShopSettings:
        # empty for now
        return ShopSettings()


class ShopDataDumper(TableDataDumper):
    entity_class = Shop
    table = t_shops

    @staticmethod
    def dump_settings(shop_settings: ShopSettings) -> dict:
        return {}


class ShopMapper(BaseMapper):
    name = 'shop'

    _merchant_oauth_relation = RelationDescription(
        name='merchant_oauth',
        base=t_shops,
        related=t_merchant_oauths,
        mapper_cls=MerchantOAuthDataMapper,
        base_cols=('uid', 'shop_id'),
        related_cols=('uid', 'shop_id'),
        outer_join=True,
    )

    _builder = CRUDQueries(
        base=t_shops,
        id_fields=('uid', 'shop_id'),
        mapper_cls=ShopDataMapper,
        dumper_cls=ShopDataDumper,
        related=(_merchant_oauth_relation,)
    )

    @staticmethod
    def _map_related(row: ValuesMapping,
                     mapper: SelectableDataMapper,
                     rel_mappers: Optional[Mapping[str, SelectableDataMapper]] = None) -> Shop:
        shop: Shop = mapper(row)
        if rel_mappers:
            if 'merchant_oauth' in rel_mappers:
                merchant_oauth = rel_mappers['merchant_oauth'](row)
                shop.oauth = (
                    merchant_oauth
                    if all((merchant_oauth.uid is not None, merchant_oauth.shop_id is not None))
                    else None
                )
        return shop

    async def create(self, obj: Shop) -> Shop:
        async with self.conn.begin():
            obj.shop_id = await self._acquire_shop_id(obj.uid)
            obj.created = obj.updated = func.now()
            insert_query, mapper = self._builder.insert(obj)
            row = await self._query_one(insert_query)
            return self._map_related(row, mapper)

    async def get_default_for_merchant(
        self, uid: int,
        shop_type: ShopType = ShopType.PROD,
        for_update: bool = False,
    ) -> Shop:
        """Get default Shop for Merchant"""
        filters = Filters()
        filters['uid'] = uid
        filters['is_default'] = True
        filters['shop_type'] = shop_type

        select_query, mapper = self._builder.select(filters=filters, for_update=for_update)
        row = await self._query_one(select_query, raise_=ShopNotFound)

        return self._map_related(row, mapper)

    async def get(self, uid: int, shop_id: int, shop_type: Optional[ShopType] = None, for_update: bool = False,
                  with_merchant_oauth: bool = False) -> Shop:
        """Get Shop by its primary key (uid, shop_id)"""
        id_values: OptValuesList = (uid, shop_id)

        filters = Filters()
        filters.add_not_none('shop_type', shop_type)

        kwargs = {
            'filters': filters,
            'id_values': id_values,
            'for_update': for_update
        }

        rel_mappers: Optional[Mapping] = None
        if with_merchant_oauth:
            select_query, mapper, rel_mappers = self._builder.select_related(**kwargs)
        else:
            select_query, mapper, = self._builder.select(**kwargs)

        row = await self._query_one(select_query, raise_=ShopNotFound)

        return self._map_related(row, mapper, rel_mappers)

    async def find(
        self,
        *,
        uid: Optional[int] = None,
        shop_id: Optional[int] = None,
        is_default: Optional[bool] = None,
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        iterator: bool = False,
        order: OptStrList = None,
        with_merchant_oauth: bool = False
    ) -> AsyncIterable[Shop]:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('shop_id', shop_id)
        filters.add_not_none('is_default', is_default)

        kwargs = {
            'filters': filters,
            'limit': limit,
            'offset': offset,
            'order': order
        }

        rel_mappers: Optional[Mapping] = None

        if with_merchant_oauth:
            select_query, mapper, rel_mappers = self._builder.select_related(**kwargs)
        else:
            select_query, mapper, = self._builder.select(**kwargs)

        async for row in self._query(select_query, iterator=iterator):
            yield self._map_related(row, mapper, rel_mappers)

    async def save(self, shop: Shop) -> Shop:
        shop.updated = func.now()
        update_query, mapper = self._builder.update(shop, ignore_fields=('uid', 'shop_id', 'created'))
        row = await self._query_one(update_query)
        return self._map_related(row, mapper)

    async def delete(self, shop: Shop) -> Shop:
        assert shop.shop_id is not None, "Can not delete Shop without knowing its shop_id"
        delete_query = self._builder.delete(shop)
        return await self._query_one(delete_query)
