from typing import AsyncIterable, Optional, Tuple

from sqlalchemy import func

from sendr_aiopg.query_builder import CRUDQueries, Filters

from mail.payments.payments.core.entities.customer_subscription_transaction import CustomerSubscriptionTransaction
from mail.payments.payments.core.entities.enums import TransactionStatus
from mail.payments.payments.storage.db.tables import \
    customer_subscription_transactions as t_customer_subscription_transactions
from mail.payments.payments.storage.exceptions import CustomerSubscriptionTransactionNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class CustomerSubscriptionTransactionDataMapper(SelectableDataMapper):
    entity_class = CustomerSubscriptionTransaction
    selectable = t_customer_subscription_transactions


class CustomerSubscriptionTransactionDataDumper(TableDataDumper):
    entity_class = CustomerSubscriptionTransaction
    table = t_customer_subscription_transactions


class CustomerSubscriptionTransactionMapper(BaseMapper[CustomerSubscriptionTransaction]):
    name = 'customer_subscription_transactions'
    _builder = CRUDQueries(
        t_customer_subscription_transactions,
        id_fields=('uid', 'customer_subscription_id', 'purchase_token'),
        mapper_cls=CustomerSubscriptionTransactionDataMapper,
        dumper_cls=CustomerSubscriptionTransactionDataDumper,
    )

    async def create(self, obj: CustomerSubscriptionTransaction) -> CustomerSubscriptionTransaction:
        obj.created = obj.updated = func.now()
        query, mapper = self._builder.insert(obj)
        return mapper(await self._query_one(query))

    async def create_or_update(self,
                               obj: CustomerSubscriptionTransaction) -> Tuple[CustomerSubscriptionTransaction, bool]:
        try:
            await self.get(obj.uid, obj.customer_subscription_id, obj.purchase_token, for_update=True)
            obj.updated = func.now()
            return await self.save(obj), False
        except CustomerSubscriptionTransactionNotFound:
            return await self.create(obj), True

    async def get(self,
                  uid: int,
                  customer_subscription_id: int,
                  purchase_token: str,
                  for_update: bool = False) -> CustomerSubscriptionTransaction:
        query, mapper = self._builder.select(id_values=(uid, customer_subscription_id, purchase_token),
                                             for_update=for_update)
        return mapper(await self._query_one(query, raise_=CustomerSubscriptionTransactionNotFound))

    async def find(self,
                   uid: Optional[int] = None,
                   customer_subscription_id: Optional[int] = None,
                   limit: Optional[int] = None,
                   offset: Optional[int] = None,
                   final: Optional[bool] = None
                   ) -> AsyncIterable[CustomerSubscriptionTransaction]:
        final_statuses = TransactionStatus.FINAL_STATUSES

        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('customer_subscription_id', customer_subscription_id)
        filters.add_not_none(
            'payment_status',
            final,
            lambda field: field.in_(final_statuses) if final else field.notin_(final_statuses)
        )

        query, mapper = self._builder.select(filters=filters, limit=limit, offset=offset)
        async for row in self._query(query):
            yield mapper(row)

    async def find_final_purchase_tokens(self,
                                         uid: int,
                                         customer_subscription_id: int,
                                         limit: Optional[int] = None,
                                         offset: Optional[int] = None,
                                         ) -> AsyncIterable[str]:
        final_statuses = TransactionStatus.FINAL_STATUSES

        filters = Filters()
        filters['uid'] = uid
        filters['customer_subscription_id'] = customer_subscription_id
        filters['payment_status'] = lambda field: field.in_(final_statuses)

        query, _ = self._builder.select(filters=filters, limit=limit, offset=offset)
        query = query.with_only_columns((t_customer_subscription_transactions.c.purchase_token,))

        async for row in self._query(query):
            yield row[0]

    async def save(self, obj: CustomerSubscriptionTransaction) -> CustomerSubscriptionTransaction:
        obj.updated = func.now()
        query, mapper = self._builder.update(obj, ignore_fields=('created', 'uid'))
        return mapper(await self._query_one(query, raise_=CustomerSubscriptionTransactionNotFound))
