from typing import AsyncIterable, Optional

from sqlalchemy import delete

from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription

from mail.payments.payments.core.entities.enums import MerchantRole
from mail.payments.payments.core.entities.user_role import UserRole
from mail.payments.payments.storage.db.tables import merchants as t_merchants
from mail.payments.payments.storage.db.tables import user_roles as t_user_roles
from mail.payments.payments.storage.db.tables import users as t_users
from mail.payments.payments.storage.exceptions import UserRoleNotFound
from mail.payments.payments.storage.mappers.base import BaseMapper
from mail.payments.payments.storage.mappers.merchant import MerchantDataMapper
from mail.payments.payments.storage.mappers.user import UserDataMapper
from mail.payments.payments.utils.datetime import utcnow
from mail.payments.payments.utils.db import SelectableDataMapper, TableDataDumper


class UserRoleDataMapper(SelectableDataMapper):
    entity_class = UserRole
    selectable = t_user_roles


class UserRoleDataDumper(TableDataDumper):
    entity_class = UserRole
    table = t_user_roles


class UserRoleMapper(BaseMapper):
    name = 'user_role'
    _user_relation = RelationDescription(
        name='user',
        base=t_user_roles,
        related=t_users,
        mapper_cls=UserDataMapper,
        base_cols=('uid',),
        related_cols=('uid',),
    )
    _merchant_relation = RelationDescription(
        name='merchant',
        base=t_user_roles,
        related=t_merchants,
        mapper_cls=MerchantDataMapper,
        base_cols=('merchant_id',),
        related_cols=('merchant_id',),
    )
    _builder = CRUDQueries(
        base=t_user_roles,
        id_fields=('uid', 'merchant_id'),
        dumper_cls=UserRoleDataDumper,
        mapper_cls=UserRoleDataMapper,
        related=(_user_relation, _merchant_relation),
    )

    @staticmethod
    def _map_related(row, mapper, rel_mappers):
        user_role = mapper(row)
        if rel_mappers and 'user' in rel_mappers:
            user_role.user = rel_mappers['user'](row)
        if rel_mappers and 'merchant' in rel_mappers:
            user_role.merchant = rel_mappers['merchant'](row)
        return user_role

    async def get(self, uid: int, merchant_id: str, with_user: bool = False, for_update: bool = False) -> UserRole:
        if not with_user:
            query, mapper = self._builder.select(id_values=(uid, merchant_id))
            rel_mappers = None
        else:
            query, mapper, rel_mappers = self._builder.select_related(id_values=(uid, merchant_id),
                                                                      for_update=for_update)
        row = await self._query_one(query, raise_=UserRoleNotFound)
        return self._map_related(row, mapper, rel_mappers)

    async def find(self,
                   uid: Optional[int] = None,
                   merchant_id: Optional[str] = None,
                   role: Optional[MerchantRole] = None,
                   with_merchant: bool = False,
                   sort_by: Optional[str] = None,
                   descending: Optional[bool] = False
                   ) -> AsyncIterable[UserRole]:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('merchant_id', merchant_id)
        filters.add_not_none('role', role)
        sort_by = sort_by or 'created'
        order = ['-' + sort_by if descending else sort_by]
        if not with_merchant:
            query, mapper = self._builder.select(filters=filters, order=order)
            rel_mappers = None
        else:
            query, mapper, rel_mappers = self._builder.select_related(filters=filters, order=order)
        async for row in self._query(query):
            yield self._map_related(row, mapper, rel_mappers)

    async def create(self, user_role: UserRole) -> UserRole:
        query, mapper = self._builder.insert(user_role)
        return mapper(await self._query_one(query))

    async def save(self, obj: UserRole) -> UserRole:
        obj.updated = utcnow()
        query, mapper = self._builder.update(obj, ignore_fields=('uid', 'merchant_id', 'created'))
        return mapper(await self._query_one(query, raise_=UserRoleNotFound))

    async def delete(self, user_role: UserRole) -> UserRole:
        query = self._builder.delete(user_role)
        return await self._query_one(query)

    async def delete_by_merchant_id(self, merchant_id: str) -> None:
        query = (
            delete(t_user_roles).
            where(t_user_roles.c.merchant_id == merchant_id)
        )
        await self.conn.execute(query)
