import logging
from dataclasses import dataclass
from typing import Any, ClassVar, List, Optional, Type, cast

from aiohttp import web

from sendr_aiopg.types import EngineUnion
from sendr_qlog import LoggerContext
from sendr_qstats.http.aiohttp import get_registry_handler, get_stats_middleware
from sendr_taskqueue import BaseStorageArbiterWorker, BaseStorageWorkerApplication
from sendr_taskqueue.worker.base.monitoring.actions import get_check_handler
from sendr_taskqueue.worker.storage.action import BaseAsyncDBAction
from sendr_taskqueue.worker.storage.monitoring.actions import WorkersHealthCheckAction

from mail.payments.payments import __version__
from mail.payments.payments.api.handlers.base import BaseHandler
from mail.payments.payments.api.middlewares import middleware_logging_adapter
from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.order.clear_unhold import CoreClearUnholdOrderAction
from mail.payments.payments.core.actions.tlog.customer_subscription import ExportCustomerSubscriptionToTLogAction
from mail.payments.payments.core.actions.tlog.order import ExportOrderToTLogAction
from mail.payments.payments.core.actions.tlog.refund import ExportRefundToTLogAction
from mail.payments.payments.core.entities.enums import ModerationType, TaskType
from mail.payments.payments.http_helpers.crypto import Crypto
from mail.payments.payments.http_helpers.partner_crypto import PartnerCrypto
from mail.payments.payments.storage import Storage, StorageContext
from mail.payments.payments.storage.logbroker.factory import LogbrokerFactory
from mail.payments.payments.storage.writers import PaymentsPushers
from mail.payments.payments.taskq.workers.abandon_terminator import AbandonTerminator
from mail.payments.payments.taskq.workers.action_worker import ActionWorker
from mail.payments.payments.taskq.workers.callback_sender import CallbackSender
from mail.payments.payments.taskq.workers.customer_subscription_updater import CustomerSubscriptionUpdater
from mail.payments.payments.taskq.workers.merchant_data_updater import MerchantDataUpdater
from mail.payments.payments.taskq.workers.moderation import FastModerationProcessor, ModerationReader, ModerationWriter
from mail.payments.payments.taskq.workers.moderation_result_notifier import MerchantModerationResultNotifier
from mail.payments.payments.taskq.workers.oauth_token_updater import OAuthTokenUpdater
from mail.payments.payments.taskq.workers.refund_starter import RefundStarter
from mail.payments.payments.taskq.workers.refund_updater import RefundUpdater
from mail.payments.payments.taskq.workers.transaction_updater import TransactionUpdater
from mail.payments.payments.utils.datetime import utcnow
from mail.payments.payments.utils.db import create_configured_engine
from mail.payments.payments.utils.stats import (
    REGISTRY, failed_tasks_count_gauge, max_pending_moderation_time_gauge, pay_status_oldest_updated, queue_tasks_gauge,
    tasks_retries_gauge
)

logger = logging.getLogger(__name__)
middleware_stats = get_stats_middleware(handle_registry=REGISTRY)


@dataclass
class TaskMetric:
    task_type: TaskType
    action_name: Optional[str]
    metric_name: str


class ArbiterWorker(BaseStorageArbiterWorker):
    storage_context_cls = StorageContext
    worker_heartbeat_period = settings.WORKER_HEARTBEAT_PERIOD

    CHECK_WORKERS_ACTIVE = True
    KILL_ON_CLEANUP = True

    _max_pending_moderation_time_types: ClassVar[List[ModerationType]] = [ModerationType.MERCHANT, ModerationType.ORDER]
    _critical_task_metrics: ClassVar[List[TaskMetric]] = [
        TaskMetric(
            task_type=TaskType.START_REFUND,
            action_name=None,
            metric_name='start_refund',
        ),
        *[
            TaskMetric(
                task_type=TaskType.RUN_ACTION,
                action_name=cast(BaseAsyncDBAction, action).action_name,
                metric_name=cast(BaseAsyncDBAction, action).action_name,
            )
            for action in [
                CoreClearUnholdOrderAction,
                ExportOrderToTLogAction,
                ExportRefundToTLogAction,
                ExportCustomerSubscriptionToTLogAction
            ]
        ]
    ]

    async def count_tasks(self, storage: Storage) -> None:
        for type_ in TaskType:
            queue_tasks_gauge.labels(type_.value).observe(0)
        async for type_, count in storage.task.count_pending_by_type():
            queue_tasks_gauge.labels(type_.value).observe(count)

        for num_retries in range(settings.DEFAULT_MAX_RETRIES + 1):
            tasks_retries_gauge.labels(num_retries).observe(0)

        async for num_retries, count in storage.task.count_pending_by_retries():
            num_retries = min(num_retries, settings.DEFAULT_MAX_RETRIES)
            tasks_retries_gauge.labels(num_retries).inc(count)

        for moderation_type in self._max_pending_moderation_time_types:
            oldest_time = await storage.moderation.get_oldest_pending_moderation_created_time(moderation_type)
            pending_seconds = 0
            if oldest_time is not None:
                current_time = utcnow()
                pending_seconds = (current_time - oldest_time).total_seconds()

            max_pending_moderation_time_gauge.labels(moderation_type.value).observe(pending_seconds)

        async for pay_status, pay_status_updated_at in (
            storage.order.get_oldest_non_terminal_pay_status_updated(settings.UID_BLACK_LIST_PAY_STATUS_MONITORING)
        ):
            pay_status_delta = (utcnow() - pay_status_updated_at).total_seconds()
            pay_status_oldest_updated.labels(pay_status.value).observe(pay_status_delta)

        await self.count_critical_failed_tasks(storage)

    async def count_critical_failed_tasks(self, storage: Storage) -> None:
        for failed_task_metric in self._critical_task_metrics:
            count = await storage.task.count_failed_tasks(failed_task_metric.task_type, failed_task_metric.action_name)
            failed_tasks_count_gauge.labels(failed_task_metric.metric_name).observe(count)


class BaseWorkerApplication(BaseStorageWorkerApplication):
    debug = settings.DEBUG
    arbiter_cls = ArbiterWorker
    storage_context_cls = StorageContext
    middlewares = (middleware_stats, middleware_logging_adapter)
    sentry_dsn = settings.SENTRY_DSN
    version = __version__

    def __init__(self, *args: Any, pushers: PaymentsPushers, crypto: Optional[Crypto] = None,
                 partner_crypto: Optional[PartnerCrypto] = None, **kwargs: Any):
        super().__init__(*args, **kwargs)
        self._logger = LoggerContext(logger, {})
        self._crypto = crypto or Crypto.from_file(settings.CRYPTO_KEYS_FILE)
        self._partner_crypto = partner_crypto or PartnerCrypto.from_file(settings.PARTNER_CRYPTO_KEY_FILE)
        self._pushers = pushers
        self._lb_factory = LogbrokerFactory(self._logger)
        self.on_cleanup.append(self.close_pushers)

    @property
    def crypto(self) -> Crypto:
        return self._crypto

    @property
    def partner_crypto(self) -> PartnerCrypto:
        return self._partner_crypto

    @property
    def pushers(self) -> PaymentsPushers:
        return self._pushers

    @property
    def lb_factory(self) -> LogbrokerFactory:
        return self._lb_factory

    def add_routes(self, *args: Any) -> None:
        super().add_routes(*args)
        self.router.add_route('GET', settings.STATS_URL, get_registry_handler(handle_registry=REGISTRY), name='stats')

    def get_check_handler(self) -> Optional[Type[BaseHandler]]:
        return get_check_handler(
            self,
            {'workers-health': WorkersHealthCheckAction},
            {'workers': self.workers, 'heartbeat_alert': settings.WORKER_HEARTBEAT_ALERT}
        )

    async def close_pushers(self, app: web.Application) -> None:
        await self.pushers.close(app)

    async def open_engine(self) -> EngineUnion:
        return await create_configured_engine()


class PaymentsWorkerApplication(BaseWorkerApplication):
    workers = [
        (TransactionUpdater, settings.TRANSACTION_UPDATER_WORKERS),
        (CustomerSubscriptionUpdater, settings.CUSTOMER_SUBSCRIPTION_UPDATER_WORKERS),
        (OAuthTokenUpdater, settings.OAUTH_TOKEN_UPDATER_WORKERS),
        (CallbackSender, settings.CALLBACK_SENDER_WORKERS),
        (RefundStarter, settings.REFUND_STARTER_WORKERS),
        (MerchantModerationResultNotifier, settings.MERCHANT_MODERATION_RESULT_NOTIFIER_WORKERS),
        (RefundUpdater, settings.REFUND_UPDATER_WORKERS),
        (ActionWorker, settings.ACTION_WORKERS),
        (AbandonTerminator, settings.ABANDON_TERMINATOR_WORKERS),
        (MerchantDataUpdater, settings.MERCHANT_DATA_UPDATER_WORKERS)
    ]


class ModerationWorkerApplication(BaseWorkerApplication):
    workers = [
        (ModerationReader, settings.MODERATION_READER_WORKERS),
        (ModerationWriter, settings.MODERATION_WRITER_WORKERS),
    ]


class FastModerationWorkerApplication(BaseWorkerApplication):
    workers = (
        (FastModerationProcessor, settings.FAST_MODERATION_WORKERS),
    )
