from aiohttp import web

from sendr_utils import utcnow

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.merchant.data_update import MerchantDataUpdateAction
from mail.payments.payments.core.entities.worker import WorkerType
from mail.payments.payments.core.exceptions import CoreFailError
from mail.payments.payments.storage.exceptions import MerchantNotFound
from mail.payments.payments.taskq.workers.base import BaseWorker
from mail.payments.payments.utils.helpers import copy_context
from mail.payments.payments.utils.stats import merchant_data_updater_max_outdated


class MerchantDataUpdater(BaseWorker):
    worker_type = WorkerType.MERCHANT_DATA_UPDATER

    @copy_context
    async def process_task(self) -> bool:
        async with self.storage_context(transact=True) as storage:
            try:
                merchant = await storage.merchant.get_for_data_update(
                    older_than_secs=settings.MERCHANT_DATA_UPDATE_DELAY_SECS,
                    locked_older_than_secs=settings.MERCHANT_DATA_OUTDATE_SECS
                )
            except MerchantNotFound:
                return False
            # mark that merchant data is updating
            merchant.data_locked = True
            await storage.merchant.save(merchant)

        try:
            merchant = await MerchantDataUpdateAction(merchant=merchant).run()
        except Exception:
            return False

        async with self.storage_context(transact=True) as storage:
            # reload merchant from DB
            try:
                db_merchant = await storage.merchant.get(uid=merchant.uid, for_update=True)
            except MerchantNotFound as exc:
                self.logger.exception('Failed fetch merchant for data update')
                raise CoreFailError from exc

            if db_merchant.data_updated_at > merchant.data_updated_at:
                # db version has more fresh data
                return True

            db_merchant.data = merchant.data
            db_merchant.data_updated_at = utcnow()
            db_merchant.data_locked = False
            await storage.merchant.save(db_merchant)
        return True

    async def update_stats(self, _: web.Application) -> None:
        async with self.storage_context() as storage:
            try:
                merchant = await storage.merchant.get_for_data_update_stats()
            except MerchantNotFound:
                merchant_data_updater_max_outdated.set(0)  # no outdated merchants
                return
            merchant_data_updater_max_outdated.set((utcnow() - merchant.data_updated_at).seconds)
