from mail.payments.payments.core.actions.worker.moderation import (
    ProcessFastModerationRequestAction, StartMerchantModerationAction, StartOrderModerationWorkerAction,
    StartSubscriptionModerationWorkerAction, UpdateModerationAction
)
from mail.payments.payments.core.entities.enums import ModerationType, TaskType, WorkerType
from mail.payments.payments.storage.logbroker.consumers.moderation import (
    FastModerationRequestConsumer, ModerationConsumer
)
from mail.payments.payments.storage.logbroker.exceptions import ProducerStartFailedException, ProducerWriteException
from mail.payments.payments.taskq.workers.base import BaseWorker
from mail.payments.payments.utils.stats import pending_moderation_gauge


class ModerationReader(BaseWorker):
    worker_type = WorkerType.MODERATION_READER

    async def process_task(self):
        consumer = ModerationConsumer(self.app.lb_factory)
        await consumer.run()
        try:
            async for moderation_result in consumer.read():
                await UpdateModerationAction(moderation_result=moderation_result).run()
        finally:
            await consumer.close()
        return False


class ModerationWriter(BaseWorker):
    worker_type = WorkerType.MODERATION_WRITER
    retry_exceptions = (ProducerStartFailedException, ProducerWriteException)

    task_action_mapping = {
        TaskType.START_MODERATION: StartMerchantModerationAction,
        TaskType.START_ORDER_MODERATION: StartOrderModerationWorkerAction,
        TaskType.START_SUBSCRIPTION_MODERATION: StartSubscriptionModerationWorkerAction,
    }

    async def update_stats(self, _):
        for type_ in ModerationType:
            pending_moderation_gauge.labels(type_.value).observe(0)

        async with self.storage_context() as storage:
            async for type_, count in storage.moderation.count_pending_by_type(ignore=False):
                pending_moderation_gauge.labels(type_.value).observe(count)


class FastModerationProcessor(BaseWorker):
    worker_type = WorkerType.FAST_MODERATION_PROCESSOR

    async def process_task(self):
        consumer = FastModerationRequestConsumer(self.app.lb_factory)
        await consumer.run()
        try:
            async for moderation_request in consumer.read():
                await ProcessFastModerationRequestAction(request=moderation_request).run()
        finally:
            await consumer.close()
        return False
