from aiohttp import web

from mail.payments.payments.core.actions.merchant.oauth_refresh import OAuthRefreshMerchantAction
from mail.payments.payments.core.entities.merchant_oauth import MerchantOAuth
from mail.payments.payments.core.entities.worker import WorkerType
from mail.payments.payments.core.exceptions import OAuthCodeError, OAuthInvalidGrants
from mail.payments.payments.utils.const import SECONDS_IN_YEAR
from mail.payments.payments.utils.helpers import copy_context
from mail.payments.payments.utils.stats import oauth_token_for_refresh_amount, oauth_token_min_remain_to_expire

from .base import BaseWorker


class OAuthTokenUpdater(BaseWorker):
    worker_type = WorkerType.OAUTH_TOKEN_UPDATER

    @copy_context
    async def process_task(self) -> bool:
        async with self.storage_context(transact=True) as storage:
            try:
                merchant_oauth: MerchantOAuth = await storage.merchant_oauth.get_for_refresh(for_update=True)
            except MerchantOAuth.DoesNotExist:
                return False

            # str(uid) для совместимости со старым форматом логов: раньше в merchant_oauth лежал merchant_id
            self.logger.context_push(merchant_id=str(merchant_oauth.uid), uid=merchant_oauth.uid)

            try:
                OAuthRefreshMerchantAction.context.storage = storage
                await OAuthRefreshMerchantAction(merchant_oauth=merchant_oauth).run()
            except OAuthInvalidGrants:
                self.logger.warning('OAuth token was revoked, disable polling')
                merchant_oauth.poll = False  # убираем токен из обновления в случае отзыва
                await storage.merchant_oauth.save(merchant_oauth)
            except OAuthCodeError:
                self.logger.error('Error OAuth token refresh')
                await storage.merchant_oauth.save(merchant_oauth)  # сохраняем в любом случае для обновления updated

            return True

    async def update_stats(self, _: web.Application) -> None:
        async with self.storage_context() as storage:
            try:
                merchant_oauth: MerchantOAuth = await storage.merchant_oauth.get_for_refresh(safe=False)
                oauth_token_min_remain_to_expire.set(merchant_oauth.expires_in)
            except MerchantOAuth.DoesNotExist:
                # Если нет токенов, считаем, что обновление должно произойти далеко в будущем
                oauth_token_min_remain_to_expire.set(2 * SECONDS_IN_YEAR)
            oauth_token_for_refresh_amount.set(await storage.merchant_oauth.count_for_refresh(safe=False))
