from datetime import timedelta

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.update_refund import UpdateRefundAction
from mail.payments.payments.core.entities.enums import RefundStatus, WorkerType
from mail.payments.payments.storage.exceptions import OrderNotFound
from mail.payments.payments.taskq.workers.base import BaseWorker
from mail.payments.payments.utils.helpers import copy_context
from mail.payments.payments.utils.stats import open_refunds_gauge


class RefundUpdater(BaseWorker):
    worker_type = WorkerType.REFUND_UPDATER

    async def update_stats(self, _):
        for status in RefundStatus:
            open_refunds_gauge.labels(status.value).observe(0)

        async with self.storage_context() as storage:
            async for status, count in storage.order.count_unfinished_refunds():
                open_refunds_gauge.labels(status.value).observe(count)

    @copy_context
    async def process_task(self):
        async with self.storage_context(transact=True) as storage:
            try:
                refund = await storage.order.get_unfinished_refund(
                    delay=timedelta(seconds=settings.REFUND_UPDATER_DELAY)
                )
            except OrderNotFound:
                return False
            UpdateRefundAction.context.storage = storage
            await UpdateRefundAction(refund=refund).run()
            return True
