import asyncio
import logging
import uuid
from importlib import import_module
from typing import Any, Dict, Type, Union

import uvloop

from sendr_qlog import LoggerContext

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.file_storage import FileStorage
from mail.payments.payments.http_helpers.crypto import Crypto
from mail.payments.payments.interactions import InteractionClients
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.storage import Storage
from mail.payments.payments.storage.logbroker.factory import LogbrokerFactory
from mail.payments.payments.storage.writers import PaymentsPushers
from mail.payments.payments.utils.db import create_configured_engine
from mail.payments.payments.utils.logging import configure_logging

ACTION_PATH_PREFIX = 'mail.payments.payments.core.actions'


def load_action_cls(action_path: str) -> Type[BaseAction]:
    if not action_path.startswith(ACTION_PATH_PREFIX):
        action_path = ACTION_PATH_PREFIX + '.' + action_path
    module_path, action_name = action_path.rsplit('.', maxsplit=1)
    module = import_module(module_path)
    return getattr(module, action_name)


def setup_context(db_engine, logger=None, loop=None):
    if logger is None:
        logger = LoggerContext(logging.getLogger(), {})
    request_id = 'manual_' + uuid.uuid4().hex
    logger.context_push(request_id=request_id)

    BaseAction.context.logger = logger
    BaseAction.context.crypto = Crypto.from_file(settings.CRYPTO_KEYS_FILE)
    BaseAction.context.request_id = request_id
    BaseAction.context.db_engine = db_engine
    BaseAction.context.pushers = PaymentsPushers(loop=loop)
    BaseAction.context.lb_factory = LogbrokerFactory(logger)


def action_command(func):
    """
    Example:
    ```
    @click.command()
    @click.argument('service_id', type=int)
    @action_command
    async def cli(context, service_id):
        context['service_id'] = service_id
        await SomeAction(context).run()
    ```
    """
    configure_logging()

    def _inner(**kwargs):
        asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())
        loop = asyncio.get_event_loop()

        async def _runner():
            db_engine = await create_configured_engine()
            logger = LoggerContext(logging.getLogger(), {})
            setup_context(db_engine=db_engine, logger=logger, loop=loop)

            try:
                return await func(**kwargs)
            finally:
                await AbstractInteractionClient.close_connector()
                await BaseAction.context.pushers.close(None)

        return loop.run_until_complete(_runner())

    return _inner


async def shell_run_action(action_class: Union[Type[BaseAction], str], **kwargs: Any) -> Any:
    """
    1. ./manage.py shell
    2. await shell_run_action('interactions.trust.GetSubscriptionInTrustAction', order=order)
    """
    configure_logging()

    db_engine = await create_configured_engine()
    async with db_engine:
        setup_context(db_engine=db_engine)
        if isinstance(action_class, str):
            action_class = load_action_cls(action_class)
        return await action_class(**kwargs).run()  # type: ignore


def create_shell_context() -> Dict[str, Any]:
    loop = asyncio.get_event_loop()

    db_engine = loop.run_until_complete(create_configured_engine())
    connection = loop.run_until_complete(db_engine.acquire())
    logger = LoggerContext(logging.getLogger(__name__), {})
    storage = Storage(connection)
    clients = InteractionClients(logger, 'manual')
    file_storage = FileStorage()

    setup_context(db_engine=db_engine, logger=logger)

    return {
        'db_engine': db_engine,
        'storage': storage,
        'logger': logger,
        'clients': clients,
        'settings': settings,
        'file_storage': file_storage,
        'shell_run_action': shell_run_action,
    }
