import os
from base64 import b64decode, b64encode

from Crypto.Cipher import AES

from mail.payments.payments.conf import settings

BLOCK_SIZE = settings.ENCRYPTION_BLOCK_SIZE
ENCRYPT_MODE = AES.MODE_CBC


def _pad(s: str) -> str:
    return s + (BLOCK_SIZE - len(s) % BLOCK_SIZE) * chr(BLOCK_SIZE - len(s) % BLOCK_SIZE)


def _unpad(s: bytes) -> bytes:
    return s[:-ord(s[len(s) - 1:])]


def _get_encryption_key(version: int) -> str:
    if version not in settings.ENCRYPTION_KEY:
        raise RuntimeError(f'Unknown key version: {version}')
    key = settings.ENCRYPTION_KEY[version]

    assert key, 'ENCRYPTION_KEY can\'t be None'
    assert len(key) in [16, 24, 32], 'ENCRYPTION_KEY must be either 16, 24, or 32 bytes long'

    return key


def encrypt(text: str, version: int) -> str:
    key = _get_encryption_key(version)
    padded = _pad(text)
    iv = os.urandom(BLOCK_SIZE)
    enc = AES.new(key, ENCRYPT_MODE, iv)
    return b64encode(iv + enc.encrypt(padded)).decode('ascii')


def decrypt(crypto_text: str, version: int) -> str:
    key = _get_encryption_key(version)
    cipher = b64decode(crypto_text.encode())
    iv = cipher[:BLOCK_SIZE]
    dec = AES.new(key, ENCRYPT_MODE, iv)
    return _unpad(dec.decrypt(cipher[BLOCK_SIZE:])).decode('ascii')
