from dataclasses import asdict
from datetime import datetime  # noqa
from enum import Enum

from sqlalchemy import text
from sqlalchemy.sql.elements import TextClause

from sendr_aiopg.data_mapper import SelectableDataMapper  # noqa
from sendr_aiopg.data_mapper import TableDataDumper as BaseTableDataDumper
from sendr_aiopg.engine.single import create_engine
from sendr_aiopg.storage import Entity

from mail.payments.payments.conf import settings


def get_db_configuration() -> dict:
    return {
        'database': settings.DATABASE['NAME'],
        'user': settings.DATABASE['USER'],
        'password': settings.DATABASE['PASSWORD'],
        'host': settings.DATABASE['HOST'],
        'port': settings.DATABASE['PORT'],
        'sslmode': 'require' if settings.DATABASE.get('USE_SSL') else 'disable',
        'connect_timeout': settings.DATABASE.get('CONNECT_TIMEOUT', 2),
        'timeout': settings.DATABASE.get('TIMEOUT', 10),  # aiopg waiter timeout
        'target_session_attrs': settings.DATABASE.get('TARGET_SESSION_ATTRS', 'read-write'),
    }


async def create_configured_engine():
    return await create_engine(**get_db_configuration())


def create_interval(seconds: int) -> TextClause:
    return text(f"INTERVAL '{seconds} SECONDS'")


class TableDataDumper(BaseTableDataDumper):
    def dump_dataclass(self, value: Entity) -> dict:
        return asdict(value, dict_factory=self._json_dict)

    @classmethod
    def _default_dump(cls, value):
        if isinstance(value, Enum):
            return value
        return super()._default_dump(value)
