import asyncio
import codecs
import csv
import re
from contextlib import contextmanager
from decimal import Decimal
from functools import wraps
from hashlib import md5
from io import StringIO
from typing import Any, AsyncGenerator, Callable, ContextManager, Iterable, List, Optional, Tuple, Type, TypeVar, Union
from uuid import uuid4

from transliterate import detect_language, translit

from sendr_qlog import LoggerContext

OptExceptionTuple = Union[Tuple[()], Tuple[Type[Exception], ...]]
_T = TypeVar('_T')

_camelcase_regex = re.compile('([A-Z])')
_underscore_regex = re.compile('_([a-z])')


def any_not_none(*i: Any) -> bool:
    return any((e is not None for e in i))


def without_none(d: dict) -> dict:
    return {
        k: (without_none(v) if isinstance(v, dict) else v)
        for k, v in d.items()
        if v is not None
    }


def masked_exception(mask_exc: Union[Type[Exception], Exception],
                     logger: Optional[LoggerContext] = None,
                     ignore_exceptions: OptExceptionTuple = ()) -> ContextManager[None]:
    """
    Masks exception with `mask_exc`. Logs original exception if logger is given
    """

    @contextmanager
    def wrapper():
        try:
            yield
        except Exception as e:
            if logger is not None:
                if isinstance(e, ignore_exceptions):
                    logger.debug('Expected masked exception', exc_info=True)
                else:
                    logger.exception('Unexpected masked exception')
            raise mask_exc

    return wrapper()


def print_done(*args: Any, **kwargs: Any) -> ContextManager[None]:
    @contextmanager
    def wrapper():
        kwargs['end'] = '... '
        print(*args, **kwargs)
        yield
        print('done')

    return wrapper()


def convert_keys(obj, convert_str, propagate=(list, dict)):
    if isinstance(obj, str):
        return convert_str(obj)
    if isinstance(obj, list):
        return [
            convert_keys(item, convert_str) if isinstance(item, propagate) else item
            for item in obj
        ]
    if isinstance(obj, dict):
        return {
            convert_str(key): convert_keys(value, convert_str) if isinstance(value, propagate) else value
            for key, value in obj.items()
        }
    return obj


def str_to_underscore(s: str) -> str:
    return _camelcase_regex.sub(lambda match: '_' + match.group(1).lower(), s)


def str_to_camelcase(s: str) -> str:
    return _underscore_regex.sub(lambda match: match.group(1).upper(), s)


def convert_to_underscore(obj):
    return convert_keys(obj, str_to_underscore)


def convert_to_camelcase(obj):
    return convert_keys(obj, str_to_camelcase)


def only_keys(d: dict, *keys: str) -> dict:
    return {
        key: d[key]
        for key in keys
    }


def safe_issubclass(cls, cls_info):
    try:
        return issubclass(cls, cls_info)
    except TypeError:
        return False


def split_list(lst: List, size: Optional[int] = None) -> List[List]:
    """Split given list into list of lists with given maximum size."""
    if not size or len(lst) == 0:
        return [lst]
    return [lst[i: i + size] for i in range(0, len(lst), size)]


def method_return_value_spy(cls: Type, method_name: str, mocker: Any, _async: bool = True) -> Any:
    """
    Helper function to create spy mock on class method
    Keeps original method logic untouched, but collects returned values into list

    Args:
        cls: class having method
        method_name: name of method to spy on
        mocker: pytest mocker object
        _async: await result of method call if method returns coroutine or similar

    Returns:
        list of returned values
    """

    returned = []
    method = getattr(cls, method_name)

    if _async:
        async def new_method(instance, *args, **kwargs):
            result = await method(instance, *args, **kwargs)
            returned.append(result)
            return result
    else:
        def new_method(instance, *args, **kwargs):
            result = method(instance, *args, **kwargs)
            returned.append(result)
            return result

    # same as mocker.spy
    mocker.patch.object(cls, method_name, side_effect=new_method, autospec=True)
    return returned


def md5_hex(string: str) -> str:
    return md5(string.encode('utf-8')).hexdigest()


def uuid_hex() -> str:
    return uuid4().hex


def decimal_format(value: Decimal) -> str:
    return str(value).rstrip('0').rstrip('.')


def temp_setattr(obj: Type, name: str, value: Any) -> ContextManager[None]:
    @contextmanager
    def wrapper():
        current_value = getattr(obj, name, value)
        setattr(obj, name, value)
        yield
        setattr(obj, name, current_value)

    return wrapper()


def copy_context(async_func: Callable) -> Callable:
    @wraps(async_func)
    async def _inner(*args, **kwargs):
        loop = asyncio.get_event_loop()
        task = loop.create_task(async_func(*args, **kwargs))
        return await task

    return _inner


def create_csv_writer(bom: str = codecs.BOM_UTF8.decode('utf-8')) -> Tuple[Any, StringIO]:
    output = StringIO()
    if bom:
        output.write(bom)
    writer = csv.writer(output)
    return writer, output


async def iter_rows(rows: Iterable[_T]) -> AsyncGenerator[_T, None]:
    for row in rows:
        yield row


def pick_opt_str(value: Optional[str], default: Optional[str]) -> Optional[str]:
    """ Выбирает непустое значение, если это возможно. Сохраняет пустое значение, если default не определен
    """
    return (
        default
        if value is None or (not value and default is not None)
        else value
    )


def is_entrepreneur_by_inn(inn: str) -> bool:
    return len(inn) > 10


def fio_str_full(surname: str, name: str, patronymic: Optional[str]) -> Optional[str]:
    if not name or not surname:
        return None
    return f'{surname} {name} {patronymic or ""}'.strip()


def monogram(s: Optional[str]) -> str:
    return '' if not s else s[0] + '.'


def fio_str_short(surname: str, name: str, patronymic: Optional[str]) -> Optional[str]:
    if not name or not surname:
        return None

    return f'{surname} {monogram(name)}{monogram(patronymic)}'


def transliterate_to_eng(s: str) -> str:
    assert s is not None
    lang = detect_language(s) or 'ru'
    return translit(s, lang, reversed=True)
