from sendr_qstats import AggSuff, Counter, Gauge, Histogram, IntSuff, MetricsRegistry, MetricSuffix

from mail.payments.payments.conf import settings


class IdempotentMetricsRegistry(MetricsRegistry):
    def __init__(self, ctype=None):
        super().__init__(ctype=ctype)
        self._idempotent = True

    def __enter__(self):
        self._idempotent = False

    def __exit__(self, exc_type, exc_val, tb):
        self._idempotent = True

    def register(self, metric):
        try:
            super().register(metric)
        except ValueError:
            if not self._idempotent:
                raise


REGISTRY = IdempotentMetricsRegistry(ctype=settings.get('STATS_CTYPE'))

with REGISTRY:
    ABS_MAX_SUFF = MetricSuffix(
        IntSuff.absolute.value,
        AggSuff.max.value,
        AggSuff.max.value,
        AggSuff.trnsp.value,
    )
    ABS_MIN_SUFF = MetricSuffix(
        IntSuff.absolute.value,
        AggSuff.min.value,
        AggSuff.min.value,
        AggSuff.trnsp.value,
    )

    interaction_response_time = Histogram(
        'response_time',
        buckets=settings.STATS_RESPONSE_TIME_BUCKETS,
        labelnames=('service',),
        registry=REGISTRY,
    )
    interaction_response_status = Counter(
        'response_status',
        labelnames=('service', 'status'),
        registry=REGISTRY,
    )

    interaction_method_response_time = Histogram(
        'response_method_time',
        buckets=settings.STATS_RESPONSE_TIME_BUCKETS,
        labelnames=('service', 'method'),
        registry=REGISTRY,
    )
    interaction_method_response_status = Counter(
        'response_method_status',
        labelnames=('service', 'method', 'status'),
        registry=REGISTRY,
    )

    queue_tasks_gauge = Gauge(
        'pending_tasks',
        labelnames=('task_type',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )

    tasks_retries_gauge = Gauge(
        'pending_tasks_retries',
        labelnames=('retries',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )

    open_transaction_gauge = Gauge(
        'open_transactions',
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )
    open_refunds_gauge = Gauge(
        'open_refunds',
        labelnames=('refund_status',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )

    pending_moderation_gauge = Gauge(
        'pending_moderation',
        labelnames=('moderation_type',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )
    max_pending_moderation_time_gauge = Gauge(
        'max_pending_moderation_time',
        labelnames=('moderation_type',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )
    merchant_moderation_time = Histogram(
        'merchant_moderation_time',
        buckets=settings.STATS_MERCHANT_MODERATION_TIME_BUCKETS,
        registry=REGISTRY,
    )
    order_moderation_time = Histogram(
        'order_moderation_time',
        buckets=settings.STATS_ORDER_MODERATION_TIME_BUCKETS,
        registry=REGISTRY,
    )
    subscription_moderation_time = Histogram(
        'subscription_moderation_time',
        buckets=settings.STATS_SUBSCRIPTION_MODERATION_TIME_BUCKETS,
        registry=REGISTRY,
    )

    paid_orders_amount = Gauge(
        'paid_orders',
        labelnames=('type',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )
    oauth_token_for_refresh_amount = Gauge(
        'oauth_token_for_refresh_amount',
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )
    oauth_token_min_remain_to_expire = Gauge(
        'oauth_token_min_remain_to_expire',
        suff=ABS_MIN_SUFF,
        registry=REGISTRY,
    )

    queue_tasks_counter = Counter('queue_tasks_counter', labelnames=('status',), registry=REGISTRY)
    queue_tasks_time = Histogram(
        'queue_tasks_time',
        labelnames=('worker',),
        buckets=settings.STATS_TASK_BUCKETS,
        registry=REGISTRY
    )

    failed_tasks_count_gauge = Gauge(
        'failed_tasks_count',
        labelnames=('task_type',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )

    merchant_data_updater_max_outdated = Gauge(
        'merchant_data_updater_max_outdated',
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )

    pay_status_oldest_updated = Gauge(
        'pay_status_oldest_updated',
        labelnames=('pay_status',),
        suff=ABS_MAX_SUFF,
        registry=REGISTRY,
    )
