CREATE SCHEMA IF NOT EXISTS payments;

CREATE TYPE payments.billing_info AS (
    client_id   text,
    person_id   text,
    contract_id text
);

CREATE TYPE payments.merchant_status AS ENUM (
    'new',
    'active',
    'inactive'
);

CREATE TABLE payments.merchants (
    uid     bigint PRIMARY KEY,
    created timestamptz not null default now(),
    billing payments.billing_info,
    name    text not null,
    status  payments.merchant_status not null default 'new'
);


CREATE TABLE payments.serials (
    -- PK
    uid             bigint PRIMARY KEY,
    -- Data
    next_revision   bigint not null default 1,
    next_order_id   bigint not null default 1,
    next_tx_id      bigint not null default 1,
    next_product_id bigint not null default 1,

    CONSTRAINT fk_serials_on_merchants FOREIGN KEY (uid)
        REFERENCES payments.merchants ON DELETE CASCADE
);

ALTER TABLE payments.merchants
    ADD CONSTRAINT fk_merchants_on_serials
    FOREIGN KEY (uid) REFERENCES payments.serials
    DEFERRABLE INITIALLY DEFERRED;


CREATE TYPE payments.product_status AS ENUM (
    'active',
    'inactive'
);

CREATE TYPE payments.nds AS ENUM (
    'nds_none',
    'nds_0',
    'nds_10',
    'nds_18'
);

CREATE TYPE payments.currency AS ENUM (
    -- Should reflect Trust currencies
    'RUR',
    'USD',
    'EUR'
);

CREATE TABLE payments.products (
    -- PK
    uid        bigint not null,
    product_id bigint not null,
    -- Data
    verified   boolean not null default false,
    created    timestamptz not null default now(),
    status     payments.product_status not null default 'active',
    nds        payments.nds not null,
    price      bigint not null,
    currency   payments.currency not null,
    name       text not null default '',

    CONSTRAINT pk_products PRIMARY KEY (uid, product_id),
    CONSTRAINT fk_products_on_merchants FOREIGN KEY (uid)
        REFERENCES payments.merchants ON DELETE CASCADE
);

CREATE UNIQUE INDEX uk_products_by_validation_fields
    ON payments.products (uid, name, nds, currency, price)
 WHERE status = 'active';


CREATE TYPE payments.order_kind AS ENUM (
    'pay',
    'refund'
);

CREATE TYPE payments.pay_order_status AS ENUM (
    'new',
    'in_progress',
    'rejected',
    'paid',
    -- Deprecated
    'verified'
);

CREATE TYPE payments.refund_order_status AS ENUM (
    'requested',
    'completed',
    'failed'
);


CREATE TABLE payments.orders (
    -- PK
    uid               bigint not null,
    order_id          bigint not null,
    -- FK
    original_order_id bigint,   -- for refunds
    -- Data
    revision          bigint not null,
    kind              payments.order_kind not null default 'pay',
    pay_status        payments.pay_order_status,
    refund_status     payments.refund_order_status,
    active            boolean not null default true,
    verified          boolean not null default false,

    currency          payments.currency not null,
    price             bigint not null,
    created           timestamptz not null default now(),
    closed            timestamptz,

    caption           text not null,            -- short
    description       text,                     -- long
    -- Non-merchant input, delayed
    user_email        text,
    user_description  text,

    CONSTRAINT pk_orders PRIMARY KEY (uid, order_id),
    CONSTRAINT fk_orders_on_merchants FOREIGN KEY (uid)
        REFERENCES payments.merchants ON DELETE RESTRICT, -- Should we allow cascade deletion?
    CONSTRAINT fk_orders_on_orders FOREIGN KEY (uid, original_order_id)
        REFERENCES payments.orders ON DELETE RESTRICT DEFERRABLE INITIALLY DEFERRED,

    CONSTRAINT check_refund_fk CHECK (
        kind != 'refund' OR original_order_id IS NOT NULL
    ),
    CONSTRAINT check_type_and_status CHECK (
        (kind = 'pay' and pay_status is not null and refund_status is null and original_order_id is null)
        OR
        (kind = 'refund' and pay_status is null and refund_status is not null and original_order_id is not null)
    )
);


CREATE TABLE payments.services (
    service_id bigint PRIMARY KEY,   -- internal or external ID?
    caption    text,
    data       jsonb
);


CREATE TABLE payments.service_connections (
    -- PK
    uid             bigint not null,
    order_id        bigint not null,
    service_id      bigint not null,
    -- Data
    revision        bigint not null,
    connection_info jsonb,

    CONSTRAINT pk_service_connections PRIMARY KEY (uid, order_id, service_id),
    CONSTRAINT fk_service_connections_on_orders FOREIGN KEY (uid, order_id)
        REFERENCES payments.orders ON DELETE RESTRICT,  -- Should we allow cascade deletion?
    CONSTRAINT fk_service_connections_on_services FOREIGN KEY (service_id)
        REFERENCES payments.services ON DELETE RESTRICT -- Should we allow cascade deletion?
);


CREATE TYPE payments.transaction_status AS ENUM (
    'active',
    'failed',
    'cancelled',
    -- Deprecated (will be renamed to 'held' and 'cleared')
    'expired',
    'success'
);

CREATE TABLE payments.transactions (
    -- PK
    uid      bigint not null,
    tx_id    bigint not null,
    -- FK -> orders
    order_id bigint not null,
    -- Data
    created  timestamptz not null default now(),
    status   payments.transaction_status not null default 'active',
    trust_purchase_token text, -- not null?

    CONSTRAINT pk_transactions PRIMARY KEY (uid, tx_id),
    CONSTRAINT fk_transactions_on_orders FOREIGN KEY (uid, order_id)
        REFERENCES payments.orders ON DELETE RESTRICT   -- Should we allow cascade deletion?
);

CREATE INDEX i_transactions_by_uid_order_id
    ON payments.transactions (uid, order_id);


CREATE FUNCTION payments.check_no_more_than_one_active_transaction()
RETURNS TRIGGER LANGUAGE plpgsql AS $$
BEGIN
    IF NEW.status = 'active' AND (
        SELECT count(*)
          FROM payments.transactions
         WHERE (uid, order_id) = (NEW.uid, NEW.order_id)
           AND status = 'active'
    ) > 0 THEN
        RAISE EXCEPTION 'Active transaction already exists for (uid = %s, order_id = %s)', NEW.uid, NEW.order_id;
    END IF;
    RETURN NEW;
END;
$$;

CREATE TRIGGER tg_transactions_no_more_than_one_active_transaction
    BEFORE INSERT ON payments.transactions
    FOR EACH ROW EXECUTE PROCEDURE payments.check_no_more_than_one_active_transaction();


CREATE TABLE payments.items (
    uid         bigint not null,
    order_id    bigint not null,
    product_id  bigint not null,
    amount      double precision not null,
    total_price bigint not null,

    CONSTRAINT pk_items PRIMARY KEY (uid, order_id, product_id),
    CONSTRAINT fk_items_on_orders FOREIGN KEY (uid, order_id)
        REFERENCES payments.orders ON DELETE CASCADE
        DEFERRABLE INITIALLY DEFERRED,
    CONSTRAINT fk_items_on_products FOREIGN KEY (uid, product_id)
        REFERENCES payments.products ON DELETE RESTRICT
);

CREATE INDEX i_items_by_uid_product_id
    ON payments.items(uid, product_id);

CREATE TYPE payments.order_item AS (
    nds         payments.nds,
    price       bigint,
    total_price bigint,
    amount      double precision,
    currency    payments.currency,
    product_id  bigint,
    name        text
);
