CREATE EXTENSION IF NOT EXISTS intarray;

CREATE TYPE action_source AS ENUM (
    'unknown', 'web', 'web_ics', 'web_for_staff', 'caldav', 'worker',
    'exchange', 'exchange_pull', 'exchange_synch', 'exchange_asynch',
    'mailhook', 'mail', 'internal_api', 'mobile', 'performer', 'db_repairer',
    'staff', 'kiosk', 'display', 'inviter', 'xiva', 'web_maya');
CREATE TYPE layer_user_perm AS ENUM ('admin', 'edit', 'view', 'create', 'list', 'access');
CREATE TYPE layer_type AS ENUM ('user', 'service', 'feed', 'absence', 'learning', 'vacation', 'illness', 'trip', 'maternity', 'conference_trip');
CREATE TYPE weekday AS ENUM ('sun', 'mon', 'tue', 'wed', 'thu', 'fri', 'sat');
CREATE TYPE grid_view AS ENUM ('day', 'week', 'month');
CREATE TYPE todo_view AS ENUM ('all', 'today', 'tomorrow', 'cur_week', 'next_week');
CREATE TYPE settings_language AS ENUM ('ru', 'en') ;
CREATE TYPE settings_invitation_accept AS ENUM ('manual', 'auto', 'robot');
CREATE TYPE availability AS ENUM ('available', 'maybe', 'busy');
CREATE TYPE job_status AS ENUM ('ready', 'starting', 'running', 'completed', 'failed', 'expired');
CREATE TYPE repetition_type AS ENUM ('daily', 'weekly', 'monthly_number', 'monthly_day_weekno', 'yearly');
CREATE TYPE todo_status AS ENUM ('needs_action', 'completed', 'in_process', 'cancelled');
CREATE TYPE event_type AS ENUM ('user', 'service', 'feed', 'absence', 'learning', 'vacation', 'illness', 'trip', 'maternity', 'conference_trip');
CREATE TYPE event_perm_all AS ENUM ('view', 'view_basics', 'none');
CREATE TYPE event_perm_participants AS ENUM ('edit', 'view');
CREATE TYPE event_completion AS ENUM ('not_applicable', 'incomplete', 'in_progress', 'complete');
CREATE TYPE event_priority AS ENUM ('low', 'normal', 'high');
CREATE TYPE event_decision AS ENUM ('undecided', 'no', 'maybe', 'yes');
CREATE TYPE notification_channel AS ENUM ('email', 'sms', 'svc', 'display', 'audio', 'xiva', 'panel', 'yamb');
CREATE TYPE ews_ignore_event_reason AS ENUM (
    'unsupported_repetition', 'canceled_event_not_found',
    'bad_organizer_email', 'bad_attendee_email',
    'no_event_in_exchange', 'is_a_copy');

CREATE TABLE layer
(
    id                  BIGSERIAL NOT NULL,
    creator_uid         BIGINT NOT NULL,
    creation_ts         TIMESTAMP,
    sid                 BIGINT NOT NULL DEFAULT 31,
    type                layer_type NOT NULL DEFAULT 'user',
    name                TEXT,
    private_token       TEXT,
    last_update_ts      TIMESTAMP NOT NULL DEFAULT '2000-01-01 00:00:00',
    coll_last_update_ts TIMESTAMP NOT NULL DEFAULT '2000-01-01 00:00:00',
    is_events_closed_by_default BOOLEAN NOT NULL DEFAULT TRUE,
    CONSTRAINT pk_layer_id PRIMARY KEY (id)
);

CREATE INDEX i_layer_creator_uid_type_sid ON layer (creator_uid, type, sid);
CREATE UNIQUE INDEX uk_layer_private_token ON layer (private_token) WHERE private_token IS NOT NULL;

CREATE TABLE layer_user
(
    id                   BIGSERIAL NOT NULL,
    uid                  BIGINT NOT NULL,
    layer_id             BIGINT NOT NULL,
    is_visible_in_ui     BOOLEAN NOT NULL DEFAULT TRUE,
    perm                 layer_user_perm NOT NULL,
    is_notify_changes    BOOLEAN NOT NULL DEFAULT FALSE,
    affects_availability BOOLEAN NOT NULL DEFAULT TRUE,
    css_class            TEXT,
    head_bg_color        INTEGER,
    head_fg_color        INTEGER,
    body_bg_color        INTEGER,
    CONSTRAINT pk_layer_user_id PRIMARY KEY (id),
    CONSTRAINT fk_layer_user_layer_id FOREIGN KEY (layer_id) REFERENCES layer (id)
);

CREATE INDEX i_layer_user_layer_id ON layer_user (layer_id);
CREATE UNIQUE INDEX uk_layer_user_uid_layer_id ON layer_user (uid, layer_id);

CREATE TABLE office
(
    id           BIGSERIAL NOT NULL,
    name         TEXT NOT NULL,
    name_en      TEXT NOT NULL,
    abbr         TEXT,
    city_name    TEXT,
    city_name_en TEXT,
    timezone_id  TEXT,
    is_active    BOOLEAN NOT NULL DEFAULT TRUE,
    invite_id    BIGINT,
    staff_id     BIGINT,
    center_id    INTEGER,
    domain       TEXT NOT NULL,
    CONSTRAINT pk_office_id PRIMARY KEY (id)
);

CREATE TABLE resource
(
    id                    BIGSERIAL NOT NULL,
    domain                TEXT NOT NULL,
    description           TEXT NOT NULL DEFAULT '',
    office_id             BIGINT NOT NULL,
    floor_id              BIGINT NOT NULL DEFAULT 0,
    floor_num             SMALLINT,
    exchange_name         TEXT,
    display_token         TEXT,
    display_last_ping_ts  TIMESTAMP,
    alter_name            TEXT,
    alter_name_en         TEXT,
    group_name            TEXT,
    group_name_en         TEXT,
    pos                   SMALLINT,
    seats                 SMALLINT,
    capacity              SMALLINT,
    video                 INTEGER,
    phone                 INTEGER,
    voice_conferencing    BOOLEAN NOT NULL DEFAULT FALSE,
    projector             SMALLINT NOT NULL DEFAULT 0,
    lcd_panel             SMALLINT NOT NULL DEFAULT 0,
    marker_board          BOOLEAN NOT NULL DEFAULT FALSE,
    desk                  BOOLEAN NOT NULL DEFAULT FALSE,
    guest_wifi            BOOLEAN NOT NULL DEFAULT FALSE,
    is_active             BOOLEAN NOT NULL DEFAULT TRUE,
    map_url               TEXT,
    type                  SMALLINT NOT NULL DEFAULT 0,
    access_group          SMALLINT,
    protection_message    TEXT,
    protection_message_en TEXT,
    sync_with_exchange    BOOLEAN NOT NULL DEFAULT TRUE,
    async_with_exchange   BOOLEAN NOT NULL DEFAULT TRUE,
    exists_on_staff       BOOLEAN NOT NULL DEFAULT TRUE,
    name                  TEXT,
    name_en               TEXT,
    staff_id              BIGINT DEFAULT NULL,
    CONSTRAINT pk_resource_id PRIMARY KEY (id),
    CONSTRAINT fk_resource_office_id FOREIGN KEY (office_id) REFERENCES office (id),
    CONSTRAINT unique_exchange_name UNIQUE (exchange_name),
    CONSTRAINT sync_with_existing_exchange_name_check CHECK (
        (exchange_name IS NOT NULL) OR
        (exchange_name IS NULL AND NOT sync_with_exchange AND NOT async_with_exchange)
    )
);

CREATE TABLE settings
(
    uid                   BIGINT NOT NULL,
    creation_ts           TIMESTAMP,
    is_edited             BOOLEAN NOT NULL DEFAULT FALSE,
    has_created_event     BOOLEAN NOT NULL DEFAULT FALSE,
    is_sid_added          BOOLEAN NOT NULL DEFAULT FALSE,
    no_ntf_start_ts       TIMESTAMP,
    no_ntf_end_ts         TIMESTAMP,
    no_ntf_start_tm       INTEGER,
    no_ntf_end_tm         INTEGER,
    start_weekday         weekday NOT NULL DEFAULT 'mon',
    last_auto_update_ts   TIMESTAMP,
    email                 TEXT NOT NULL,
    yandex_email          TEXT NOT NULL,
    domain                TEXT,
    user_login            TEXT,
    user_name             TEXT,
    view_type             grid_view NOT NULL DEFAULT 'week',
    todo_view_type        todo_view NOT NULL DEFAULT 'all',
    show_todo             BOOLEAN NOT NULL DEFAULT TRUE,
    timezone_javaid       TEXT NOT NULL DEFAULT 'Europe/Moscow',
    geo_tz_javaid         TEXT NOT NULL DEFAULT 'Europe/Moscow',
    grid_top_hours        SMALLINT NOT NULL DEFAULT 8,
    layer_id              BIGINT,
    translit_sms          BOOLEAN NOT NULL DEFAULT TRUE,
    maps_enabled          BOOLEAN NOT NULL DEFAULT TRUE,
    show_availability     BOOLEAN NOT NULL DEFAULT TRUE,
    show_avatars          BOOLEAN NOT NULL DEFAULT TRUE,
    language              settings_language NOT NULL DEFAULT 'ru',
    inv_accept_type       settings_invitation_accept NOT NULL DEFAULT 'manual',
    hack_caldav_timezones BOOLEAN NOT NULL DEFAULT FALSE,
    todo_planned_email_tm INTEGER,
    todo_expired_email_tm INTEGER,
    cerberus_user         BOOLEAN DEFAULT FALSE,
    CONSTRAINT pk_settings_uid PRIMARY KEY (uid)
);

CREATE INDEX i_settings_email ON settings (email);
CREATE INDEX i_settings_yandex_email ON settings (yandex_email);
CREATE INDEX i_settings_login ON settings (user_login);

CREATE TABLE settings_yt
(
    uid                      BIGINT NOT NULL,
    let_participants_edit    BOOLEAN NOT NULL,
    xiva_reminder_enabled    BOOLEAN NOT NULL,
    remind_undecided         BOOLEAN NOT NULL DEFAULT FALSE,
    no_ntf_during_absence    BOOLEAN NOT NULL DEFAULT TRUE,
    is_dismissed             BOOLEAN NOT NULL,
    table_office_id          BIGINT,
    table_floor_num          SMALLINT,
    active_office_id         BIGINT,
    active_office_detect_ts  TIMESTAMP,
    default_ics_availability availability NOT NULL DEFAULT 'available',
    is_outlooker             BOOLEAN NOT NULL DEFAULT FALSE,
    is_ewser BOOLEAN NOT NULL DEFAULT FALSE,
    CONSTRAINT pk_settings_yt_uid PRIMARY KEY (uid)
);

CREATE TABLE repetition
(
    id                  BIGSERIAL NOT NULL,
    type                repetition_type NOT NULL,
    r_each              INTEGER,
    r_weekly_days       TEXT,
    r_monthly_lastweek  BOOLEAN,
    due_ts              TIMESTAMP,
    --
    CONSTRAINT pk_repetition_id PRIMARY KEY (id)
);

CREATE INDEX i_repetition_due_ts ON repetition (due_ts);

CREATE TABLE ics_feed
(
    layer_id            BIGINT NOT NULL,
    uid                 BIGINT NOT NULL,
    url                 TEXT NOT NULL,
    creation_ts         TIMESTAMP NOT NULL,
    last_query_ts       TIMESTAMP,
    http_last_mod_ts    TIMESTAMP,
    http_etag           TEXT,
    last_success_ts     TIMESTAMP,
    useless_attempts    INTEGER NOT NULL DEFAULT 0,
    interval_min        INTEGER NOT NULL,
    next_query_ts       TIMESTAMP NOT NULL,
    hash                TEXT,
    CONSTRAINT pk_ics_feed_layer_id PRIMARY KEY (layer_id),
    CONSTRAINT fk_ics_feed_layer_id FOREIGN KEY (layer_id) REFERENCES layer (id)
);

CREATE INDEX i_ics_feed_uid ON ics_feed (uid);
CREATE INDEX i_ics_feed_next_query_ts ON ics_feed (next_query_ts);

CREATE TABLE todo_list
(
    id                  BIGSERIAL NOT NULL,
    creator_uid         BIGINT NOT NULL,
    creation_ts         TIMESTAMP NOT NULL,
    creation_source     action_source NOT NULL DEFAULT 'unknown',
    title               TEXT NOT NULL,
    description         TEXT NOT NULL DEFAULT '',
    deleted             BOOLEAN NOT NULL DEFAULT FALSE,
    deletion_ts         TIMESTAMP,
    deletion_source     action_source NOT NULL DEFAULT 'unknown',
    last_update_ts      TIMESTAMP NOT NULL DEFAULT '1970-01-01 00:00:00',
    modification_source action_source NOT NULL DEFAULT 'unknown',
    color               INTEGER,
    external_id         TEXT NOT NULL,
    caldav_coll_id      TEXT,
    CONSTRAINT pk_todo_list_id PRIMARY KEY (id)
);

CREATE INDEX i_todo_list_creator_uid_creation_ts ON todo_list (creator_uid, creation_ts);

CREATE TABLE todo_item
(
    id                        BIGSERIAL NOT NULL,
    todo_list_id              BIGINT NOT NULL,
    creation_ts               TIMESTAMP NOT NULL,
    creation_source           action_source NOT NULL DEFAULT 'unknown',
    title                     TEXT NOT NULL,
    description               TEXT NOT NULL DEFAULT '',
    location                  TEXT NOT NULL DEFAULT '',
    start_ts                  TIMESTAMP,
    is_all_day                BOOLEAN NOT NULL DEFAULT FALSE,
    complete_percent          SMALLINT,
    priority                  SMALLINT,
    status                    todo_status,
    completion_ts             TIMESTAMP,
    due_ts                    TIMESTAMP,
    pos                       INTEGER NOT NULL,
    timestamp_position        BIGINT NOT NULL DEFAULT 0,
    ical_x_apple_sort_order   BIGINT,
    color                     INTEGER,
    sequence                  INTEGER NOT NULL DEFAULT 0,
    last_update_ts            TIMESTAMP NOT NULL DEFAULT '1970-01-01 00:00:00',
    modification_source       action_source NOT NULL DEFAULT 'unknown',
    external_id               TEXT NOT NULL,
    deleted                   BOOLEAN NOT NULL DEFAULT FALSE,
    deletion_ts               TIMESTAMP,
    deletion_source           action_source NOT NULL DEFAULT 'unknown',
    archived                  BOOLEAN NOT NULL DEFAULT FALSE,
    archived_ts               TIMESTAMP,
    is_today                  BOOLEAN NOT NULL DEFAULT FALSE,
    is_this_week              BOOLEAN NOT NULL DEFAULT FALSE,
    mail_message_id           TEXT,
    mail_subject              TEXT,
    ical_geo_notification     TEXT,
    notification_ts           TIMESTAMP,
    notification_sent_ts      TIMESTAMP,
    notification_shown_in_web BOOLEAN NOT NULL DEFAULT FALSE,
    ical_x_moz_snooze_time    TIMESTAMP,
    ical_x_moz_lastack        TIMESTAMP,
    creator_uid BIGINT,
    CONSTRAINT pk_todo_item_id PRIMARY KEY (id),
    CONSTRAINT fk_todo_item_todo_list_id FOREIGN KEY (todo_list_id) REFERENCES todo_list (id)
);

CREATE INDEX i_todo_item_todo_list_id ON todo_item (todo_list_id);
CREATE INDEX i_todo_item_external_id ON todo_item (external_id);
CREATE INDEX i_todo_item_notification_ts ON todo_item (notification_ts);
CREATE INDEX i_todo_item_creator_uid_completion_ts_due_ts ON todo_item (creator_uid, completion_ts, due_ts);

CREATE TABLE main_event
(
    id                     BIGSERIAL NOT NULL,
    external_id            TEXT NOT NULL,
    external_id_normalized TEXT NOT NULL DEFAULT '',
    timezone_id            TEXT NOT NULL,
    last_update_ts         TIMESTAMP NOT NULL DEFAULT '2000-01-01 00:00:00',
    is_exported_with_ews   BOOLEAN,
    CONSTRAINT pk_main_event_id PRIMARY KEY (id)
);

CREATE INDEX i_main_event_external_id ON main_event (external_id);
CREATE INDEX i_main_event_external_id_normalized ON main_event (external_id_normalized);

CREATE TABLE event
(
    id                  BIGSERIAL NOT NULL,
    creator_uid         BIGINT NOT NULL,
    type                event_type NOT NULL,
    sid                 BIGINT NOT NULL DEFAULT 31,
    is_all_day          BOOLEAN NOT NULL DEFAULT FALSE,
    start_ts            TIMESTAMP NOT NULL,
    end_ts              TIMESTAMP NOT NULL,
    name                TEXT NOT NULL,
    location            TEXT NOT NULL DEFAULT '',
    description         TEXT NOT NULL DEFAULT '',    
    url                 TEXT,
    main_event_id       BIGINT NOT NULL,
    repetition_id       BIGINT,
    recurrence_id       TIMESTAMP,
    creation_ts         TIMESTAMP NOT NULL,
    last_update_ts      TIMESTAMP NOT NULL,
    perm_all            event_perm_all NOT NULL DEFAULT 'view',
    perm_participants   event_perm_participants NOT NULL DEFAULT 'view',
    participants_invite BOOLEAN NOT NULL DEFAULT FALSE,
    sequence            INTEGER NOT NULL DEFAULT 0,
    dtstamp             TIMESTAMP,
    creation_source     action_source NOT NULL DEFAULT 'unknown',
    modification_source action_source NOT NULL DEFAULT 'unknown',
    creation_req_id     TEXT,
    modification_req_id TEXT,
    data                JSONB DEFAULT NULL,
    CONSTRAINT pk_event_id PRIMARY KEY (id),
    CONSTRAINT fk_event_main_event_id FOREIGN KEY (main_event_id) REFERENCES main_event (id),
    CONSTRAINT fk_event_repetition_id FOREIGN KEY (repetition_id) REFERENCES repetition (id) ON DELETE SET NULL
);

CREATE INDEX i_event_repetition_id ON event (repetition_id);
CREATE INDEX i_event_end_ts_start_ts_repetition_id ON event (end_ts, start_ts, repetition_id);
CREATE INDEX i_event_start_ts_end_ts_repetition_id ON event (start_ts, end_ts, repetition_id);
CREATE INDEX i_event_creator_uid_is_all_day ON event (creator_uid, is_all_day);
CREATE UNIQUE INDEX uk_event_main_event_id_null_recurrence_id
    ON event (main_event_id) WHERE recurrence_id IS NULL;
CREATE UNIQUE INDEX uk_event_main_event_id_recurrence_id ON event (main_event_id, recurrence_id);

CREATE TABLE event_layer
(
    event_id             BIGINT NOT NULL,
    layer_id             BIGINT NOT NULL,
    is_primary_inst      BOOLEAN NOT NULL DEFAULT FALSE,
    l_creator_uid        BIGINT NOT NULL,
    creation_source      action_source NOT NULL DEFAULT 'unknown',
    modification_source  action_source NOT NULL DEFAULT 'unknown',
    creation_req_id      TEXT,
    modification_req_id  TEXT,
    CONSTRAINT pk_event_layer_event_id_layer_id PRIMARY KEY (event_id, layer_id),
    event_start_ts TIMESTAMP NOT NULL,
    event_end_ts TIMESTAMP NOT NULL,
    repetition_due_ts TIMESTAMP,
    CONSTRAINT fk_event_layer_event_id FOREIGN KEY (event_id) REFERENCES event (id),
    CONSTRAINT fk_event_layer_layer_id FOREIGN KEY (layer_id) REFERENCES layer (id)
);

CREATE INDEX i_event_layer_layer_id_event_id ON event_layer (layer_id, event_id);
CREATE INDEX i_event_layer_is_primary_inst_event_id ON event_layer (is_primary_inst, event_id) WHERE is_primary_inst = TRUE;
CREATE INDEX i_event_layer_l_creator_uid_event_id ON event_layer (l_creator_uid, event_id);
CREATE INDEX i_event_layer_layer_id_event_end_ts_event_start_ts
    ON event_layer (layer_id, event_end_ts, event_start_ts)
    WHERE repetition_due_ts IS NULL;
CREATE INDEX i_event_layer_layer_id_repetition_due_ts_event_start_ts
    ON event_layer (layer_id, repetition_due_ts, event_start_ts)
    WHERE repetition_due_ts IS NOT NULL;

CREATE TABLE event_resource
(
    event_id             BIGINT NOT NULL,
    resource_id          BIGINT NOT NULL,
    exchange_id          TEXT,
    creation_source      action_source NOT NULL DEFAULT 'unknown',
    modification_source  action_source NOT NULL DEFAULT 'unknown',
    creation_req_id      TEXT,
    modification_req_id  TEXT,
    CONSTRAINT pk_event_resource_event_id_resource_id PRIMARY KEY (event_id, resource_id),
    event_start_ts TIMESTAMP NOT NULL,
    event_end_ts TIMESTAMP NOT NULL,
    repetition_due_ts TIMESTAMP,
    CONSTRAINT fk_event_resource_event_id FOREIGN KEY (event_id) REFERENCES event (id),
    CONSTRAINT fk_event_resource_resource_id FOREIGN KEY (resource_id) REFERENCES resource (id)
);

CREATE INDEX i_event_resource_resource_id_event_id ON event_resource (resource_id, event_id);
CREATE UNIQUE INDEX uk_event_resource_exchange_id ON event_resource (exchange_id) WHERE exchange_id IS NOT NULL;
CREATE INDEX i_event_resource_resource_id_event_end_ts_event_start_ts
    ON event_resource (resource_id, event_end_ts, event_start_ts)
    WHERE repetition_due_ts IS NULL;
CREATE INDEX i_event_resource_resource_id_repetition_due_ts_event_start_ts
    ON event_resource (resource_id, repetition_due_ts, event_start_ts)
    WHERE repetition_due_ts IS NOT NULL;

CREATE TABLE event_user
(
    id                     BIGSERIAL NOT NULL,
    event_id               BIGINT NOT NULL,
    uid                    BIGINT NOT NULL,
    last_sent_ts           TIMESTAMP,
    last_sent_sms_id       TEXT,
    total_sent_count       INTEGER NOT NULL DEFAULT 0,
    availability           availability NOT NULL DEFAULT 'available',
    completion             event_completion NOT NULL DEFAULT 'not_applicable',
    priority               event_priority NOT NULL DEFAULT 'normal',
    decision               event_decision NOT NULL DEFAULT 'yes',
    is_attendee            BOOLEAN NOT NULL DEFAULT FALSE,
    is_organizer           BOOLEAN NOT NULL DEFAULT FALSE,
    is_subscriber          BOOLEAN NOT NULL DEFAULT FALSE,
    private_token          TEXT,
    reason                 TEXT,
    sequence               INTEGER NOT NULL DEFAULT 0,
    dtstamp                TIMESTAMP,
    last_update            TIMESTAMP NOT NULL DEFAULT '2000-01-01 00:00:00',
    exchange_id            TEXT,
    creation_source        action_source NOT NULL DEFAULT 'unknown',
    modification_source    action_source NOT NULL DEFAULT 'unknown',
    creation_req_id        TEXT,
    modification_req_id    TEXT,
    ical_x_moz_snooze_time TIMESTAMP,
    ical_x_moz_lastack     TIMESTAMP,
    decision_source action_source,
    CONSTRAINT pk_event_user_id PRIMARY KEY (id),
    CONSTRAINT fk_event_user_event_id FOREIGN KEY (event_id) REFERENCES event (id)
);

CREATE INDEX i_event_user_event_id ON event_user (event_id);
CREATE INDEX i_event_user_private_token ON event_user (private_token) WHERE private_token IS NOT NULL;
CREATE UNIQUE INDEX uk_event_user_uid_event_id ON event_user (uid, event_id);
CREATE UNIQUE INDEX uk_event_user_exchange_id ON event_user (exchange_id) WHERE exchange_id IS NOT NULL;

CREATE TABLE event_attachment
(
    id                  BIGSERIAL NOT NULL,
    filename            TEXT NOT NULL,
    external_id         TEXT NOT NULL,
    main_event_id       BIGINT NOT NULL,
    CONSTRAINT pk_event_attachment_id PRIMARY KEY (id),
    CONSTRAINT fk_event_attachment_main_event_id FOREIGN KEY (main_event_id) REFERENCES main_event (id)
);

CREATE INDEX i_event_attachment_main_event_id ON event_attachment (main_event_id);

CREATE TABLE event_timezone_info
(
    main_event_id       BIGINT NOT NULL,
    original_ics_id     TEXT,
    CONSTRAINT pk_event_timezone_info_main_event_id PRIMARY KEY (main_event_id),
    CONSTRAINT fk_event_timezone_info_main_event_id FOREIGN KEY (main_event_id) REFERENCES main_event (id)
);

CREATE TABLE repetition_confirmation
(
    main_event_id    BIGINT NOT NULL,
    deadline         TIMESTAMP NOT NULL,
    next_query_ts    TIMESTAMP NOT NULL,
    CONSTRAINT pk_repetition_confirmation_main_event_id PRIMARY KEY (main_event_id),
    CONSTRAINT fk_repetition_confirmation_main_event_id FOREIGN KEY (main_event_id) REFERENCES main_event (id)
);

CREATE INDEX ON repetition_confirmation (next_query_ts);

CREATE TABLE event_resource_uncheckin
(
    main_event_id   BIGINT NOT NULL,
    resource_id     BIGINT NOT NULL,
    previous_date   DATE NOT NULL,
    times_in_a_row  SMALLINT NOT NULL,
    CONSTRAINT pk_event_resource_uncheckin_main_event_id_resource_id PRIMARY KEY (main_event_id, resource_id)
);

CREATE TABLE resource_reservation
(
    creator_uid     BIGINT NOT NULL,
    reservation_id  BIGINT NOT NULL,
    resource_id     BIGINT NOT NULL,
    deadline        TIMESTAMP NOT NULL,
    start_ts        TIMESTAMP NOT NULL,
    end_ts          TIMESTAMP NOT NULL,
    timezone_id     TEXT NOT NULL,
    r_rule          TEXT NULL,
    r_due_ts        TIMESTAMP NULL,
    CONSTRAINT pk_resource_reservation_creator_uid_reservation_id_resource_id PRIMARY KEY (creator_uid, reservation_id, resource_id)
);

CREATE INDEX i_resource_reservation_resource_id_deadline ON resource_reservation (resource_id, deadline);

CREATE TABLE event_notification
(
    id                      BIGSERIAL NOT NULL,
    channel                 notification_channel NOT NULL,
    offset_minute           INTEGER NOT NULL,
    next_send_ts            TIMESTAMP,
    event_user_id           BIGINT NOT NULL,
    CONSTRAINT pk_event_notification_id PRIMARY KEY (id),
    CONSTRAINT fk_event_notification_event_user_id FOREIGN KEY (event_user_id) REFERENCES event_user (id)    
);

CREATE INDEX i_event_notification_event_user_id ON event_notification (event_user_id);
CREATE INDEX i_event_notification_next_send_ts ON event_notification (next_send_ts);

CREATE TABLE layer_notification
(
    id                      BIGSERIAL NOT NULL,
    channel                 notification_channel NOT NULL,
    offset_minute           INTEGER NOT NULL,
    layer_user_id           BIGINT NOT NULL,
    CONSTRAINT pk_layer_notification_id PRIMARY KEY (id),
    CONSTRAINT fk_layer_notification_layer_user_id FOREIGN KEY (layer_user_id) REFERENCES layer_user (id)
);

CREATE INDEX i_layer_notification_layer_user_id ON layer_notification (layer_user_id);

CREATE TABLE sending_sms
(
    uid               BIGINT NOT NULL,
    active_id         TEXT NOT NULL,
    message           TEXT NOT NULL,
    submit_ts         TIMESTAMP NOT NULL,
    submit_req_id     TEXT,
    expiration_ts     TIMESTAMP NOT NULL,
    is_processed      BOOLEAN NOT NULL DEFAULT FALSE,
    sent_ts           TIMESTAMP,
    sent_sms_id       TEXT,
    sent_fail_reason  TEXT,
    CONSTRAINT pk_sending_sms_uid_active_id PRIMARY KEY (uid, active_id)
);

CREATE INDEX i_sending_sms_not_is_processed ON sending_sms (is_processed) WHERE is_processed = FALSE;

CREATE TABLE event_invitation
(
    event_id            BIGINT NOT NULL,
    email               TEXT NOT NULL,
    creation_ts         TIMESTAMP NOT NULL,
    creator_uid         BIGINT NOT NULL,
    name                TEXT NOT NULL DEFAULT '',
    private_token       TEXT,
    decision            event_decision NOT NULL DEFAULT 'undecided',
    is_organizer        BOOLEAN NOT NULL DEFAULT FALSE,
    reason              TEXT,
    sequence            INTEGER NOT NULL DEFAULT 0,
    dtstamp             TIMESTAMP,
    creation_source     action_source NOT NULL DEFAULT 'unknown',
    modification_source action_source NOT NULL DEFAULT 'unknown',
    creation_req_id     TEXT,
    modification_req_id TEXT,
    decision_source action_source,
    CONSTRAINT pk_event_invitation_event_id_email PRIMARY KEY (event_id, email),
    CONSTRAINT fk_event_invitation_event_id FOREIGN KEY (event_id) REFERENCES event (id)
);

CREATE INDEX i_event_invitation_private_token ON event_invitation (private_token);

CREATE TABLE layer_invitation
(
    layer_id            BIGINT NOT NULL,
    email               TEXT NOT NULL,
    creation_ts         TIMESTAMP NOT NULL,
    creator_uid         BIGINT NOT NULL,
    uid                 BIGINT,
    name                TEXT NOT NULL DEFAULT '',
    private_token       TEXT,
    perm                layer_user_perm NOT NULL,
    creation_source     action_source NOT NULL DEFAULT 'unknown',
    modification_source action_source NOT NULL DEFAULT 'unknown',
    creation_req_id     TEXT,
    modification_req_id TEXT,
    CONSTRAINT pk_layer_invitation_layer_id_email PRIMARY KEY (layer_id, email),
    CONSTRAINT fk_layer_invitation_layer_id FOREIGN KEY (layer_id) REFERENCES layer (id)
);

CREATE UNIQUE INDEX uk_layer_invitation_layer_id_uid ON layer_invitation (layer_id, uid) WHERE uid IS NOT NULL;
CREATE INDEX i_layer_invitation_private_token ON layer_invitation (private_token);

CREATE TABLE rdate
(
    id                  BIGSERIAL NOT NULL,
    start_ts            TIMESTAMP NOT NULL,
    end_ts              TIMESTAMP NULL,
    event_id            BIGINT NOT NULL,
    is_rdate            BOOLEAN NOT NULL DEFAULT TRUE,
    creation_req_id     TEXT,
    creation_ts         TIMESTAMP,
    CONSTRAINT pk_rdate_id PRIMARY KEY (id),
    CONSTRAINT fk_rdate_event_id  FOREIGN KEY (event_id) REFERENCES event (id)
);

CREATE INDEX rdate_event_id ON rdate (event_id);
CREATE INDEX rdate_is_rdate ON rdate (is_rdate);

CREATE TABLE deleted_event
(
    id                        BIGINT NOT NULL,
    start_ts                  TIMESTAMP NOT NULL,
    name                      TEXT NOT NULL,
    external_id               TEXT NOT NULL,
    recurrence_id             TIMESTAMP NULL,
    deletion_source           action_source NOT NULL DEFAULT 'unknown',
    deletion_ts               TIMESTAMP NOT NULL,
    deletion_req_id           TEXT
);

CREATE INDEX i_deleted_event_id ON deleted_event (id);
CREATE INDEX i_deleted_event_external_id ON deleted_event (external_id);

CREATE TABLE deleted_event_user
(
    id                       BIGINT NOT NULL,
    event_id                 BIGINT NOT NULL,
    uid                      BIGINT NOT NULL,
    is_attendee              BOOLEAN NOT NULL DEFAULT FALSE,
    is_organizer             BOOLEAN NOT NULL DEFAULT FALSE,
    exchange_id              TEXT,
    deletion_source          action_source NOT NULL DEFAULT 'unknown',
    deletion_ts              TIMESTAMP NOT NULL,
    deletion_req_id          TEXT
);

CREATE INDEX i_deleted_event_user_id ON deleted_event_user (id);
CREATE INDEX i_deleted_event_user_uid_event_id ON deleted_event_user (uid, event_id);
CREATE INDEX i_deleted_event_user_exchange_id ON deleted_event_user (exchange_id);

CREATE TABLE deleted_event_layer
(
    event_id                 BIGINT NOT NULL,
    layer_id                 BIGINT NOT NULL,
    deletion_source          action_source NOT NULL DEFAULT 'unknown',
    deletion_ts              TIMESTAMP NOT NULL,
    deletion_req_id          TEXT
);

CREATE INDEX i_deleted_event_layer_layer_id_deletion_ts ON deleted_event_layer (layer_id, deletion_ts);
CREATE INDEX i_deleted_event_layer_event_id_layer_id ON deleted_event_layer (event_id, layer_id);

CREATE TABLE deleted_event_resource
(
    event_id                 BIGINT NOT NULL,
    resource_id              BIGINT NOT NULL,
    exchange_id              TEXT,
    deletion_source          action_source NOT NULL DEFAULT 'unknown',
    deletion_ts              TIMESTAMP NOT NULL,
    deletion_req_id          TEXT --
);

CREATE INDEX i_deleted_event_resource_event_id_resource_id ON deleted_event_resource (event_id, resource_id);
CREATE INDEX i_deleted_event_resource_exchange_id ON deleted_event_resource (exchange_id);

CREATE TABLE yt_ews_subscription
(
    id                   BIGSERIAL NOT NULL,
    uid                  BIGINT,
    resource_id          BIGINT,
    email                TEXT,
    subscription_ts      TIMESTAMP,
    subscription_id      TEXT,
    last_ping_ts         TEXT,
    pull_subscription_id TEXT,
    pull_subscription_ts TIMESTAMP,
    pull_watermark       TEXT,
    last_pull_ts         TIMESTAMP,
    CONSTRAINT pk_yt_ews_subscription PRIMARY KEY (id)
);

CREATE UNIQUE INDEX uk_yt_ews_subscription_uid ON yt_ews_subscription (uid) WHERE uid IS NOT NULL;
CREATE UNIQUE INDEX uk_yt_ews_subscription_resource_id ON yt_ews_subscription (resource_id) WHERE resource_id IS NOT NULL;
CREATE INDEX i_yt_ews_subscription_subscription_id ON yt_ews_subscription (subscription_id);

CREATE TABLE yt_ews_ignored_event
(
    exchange_id         TEXT NOT NULL,
    reason              ews_ignore_event_reason NOT NULL,
    message             TEXT NULL,
    CONSTRAINT pk_yt_ignored_event_exchange_id PRIMARY KEY (exchange_id)
);

CREATE TABLE yt_ews_org_cache
(
    exchange_id         TEXT NOT NULL,
    email               TEXT NOT NULL,
    CONSTRAINT pk_yt_ews_org_cache PRIMARY KEY (exchange_id)
);

CREATE TABLE yt_ews_exporting_event
(
    external_id          TEXT NOT NULL,
    last_submit_ts       TIMESTAMP NOT NULL,
    last_attempt_ts      TIMESTAMP,
    last_attempt_req_id  TEXT,
    failure_reason       TEXT,
    next_attempt_ts      TIMESTAMP,
    CONSTRAINT pk_yt_ews_exporting_event_external_id PRIMARY KEY (external_id)
);

CREATE TABLE resource_schedule
(
    resource_id     BIGINT NOT NULL,
    day_start       TIMESTAMP NOT NULL,
    event_intervals TEXT NOT NULL DEFAULT '',
    version         INTEGER NOT NULL DEFAULT 0,
    is_valid        BOOLEAN NOT NULL DEFAULT TRUE,
    CONSTRAINT pk_resource_schedule_resource_id_day_start PRIMARY KEY (resource_id, day_start),
    CONSTRAINT fk_resource_schedule_resource_id FOREIGN KEY (resource_id) REFERENCES resource (id)
);

CREATE TABLE resource_conflict
(
    id            BIGSERIAL NOT NULL,
    resource_id   BIGINT NOT NULL,
    external_id   TEXT NOT NULL,
    start_ts      TIMESTAMP NOT NULL,
    end_ts        TIMESTAMP NOT NULL,
    name          TEXT NOT NULL,
    exchange_id   TEXT NOT NULL,
    recurrence_id TIMESTAMP NULL,
    creation_ts   TIMESTAMP NOT NULL,
    CONSTRAINT pk_resource_conflict_id PRIMARY KEY (id),
    CONSTRAINT fk_resource_conflict_resource_id FOREIGN KEY (resource_id) REFERENCES resource (id)
);

CREATE INDEX i_resource_conflict_start_ts ON resource_conflict (start_ts);
CREATE UNIQUE INDEX uk_resource_conflict_resource_id_external_id_start_ts
    ON resource_conflict (resource_id, exchange_id, start_ts);

CREATE TABLE ui_settings
(
    uid               BIGINT NOT NULL,
    todo_web_settings TEXT NOT NULL DEFAULT '',
    CONSTRAINT pk_ui_settings PRIMARY KEY (uid)
);

CREATE TABLE user_groups
(
    id             BIGSERIAL NOT NULL,
    uid            BIGINT NULL,
    department_url TEXT NULL,    
    groups INTEGER[] NOT NULL DEFAUlT '{}',
    resources_can_access INTEGER[] NOT NULL DEFAULT '{}',
    resources_can_admin INTEGER[] NOT NULL DEFAULT '{}',
    CONSTRAINT pk_user_groups PRIMARY KEY (id),
    CONSTRAINT groups_unique_values CHECK (array_length(uniq(sort(groups)), 1) = array_length(groups, 1)),
    CONSTRAINT resources_can_access_unique CHECK (array_length(uniq(sort(resources_can_access)), 1) = array_length(resources_can_access, 1)),
    CONSTRAINT resources_can_admin_unique CHECK (array_length(uniq(sort(resources_can_admin)), 1) = array_length(resources_can_admin, 1))
);

CREATE UNIQUE INDEX uk_user_groups_uid ON user_groups (uid) WHERE uid IS NOT NULL;
CREATE UNIQUE INDEX uk_user_groups_department_url ON user_groups (department_url) WHERE department_url IS NOT NULL;

CREATE TABLE todo_list_email
(
    id            BIGSERIAL NOT NULL,
    uid           BIGINT NOT NULL,
    sent_ts       TIMESTAMP NOT NULL DEFAULT (now() AT TIME ZONE 'UTC'),
    email         TEXT NOT NULL,
    todo_list_id  BIGINT NOT NULL,
    CONSTRAINT pk_todo_list_email_id PRIMARY KEY (id)
);

CREATE INDEX i_todo_list_email_uid_sent_ts ON todo_list_email (uid, sent_ts);

CREATE TABLE todo_export_token
(
    uid           BIGINT NOT NULL,
    token         TEXT NOT NULL,
    CONSTRAINT pk_todo_export_token_uid PRIMARY KEY (uid)
);

CREATE UNIQUE INDEX uk_todo_export_token_token ON todo_export_token (token);

CREATE TABLE notification_send_stat
(
    id                BIGSERIAL NOT NULL,
    task_req_id       TEXT      NOT NULL,
    task_start_ts     TIMESTAMP NOT NULL,
    task_end_ts       TIMESTAMP NOT NULL,
    emails_sent       INT       NOT NULL,
    emails_failed     INT       NOT NULL,
    sms_sent          INT       NOT NULL,
    sms_failed        INT       NOT NULL,
    total_processed   INT       NOT NULL,
    unexpected_errors INT       NOT NULL,
    CONSTRAINT pk_notification_send_stat_id PRIMARY KEY (id)
);

CREATE INDEX i_notification_send_stat_task_start_ts ON notification_send_stat (task_start_ts);

CREATE TABLE job
(
    id UUID NOT NULL,
    task TEXT NOT NULL,
    create_time TIMESTAMP NOT NULL,
    active_uid TEXT,
    parameters JSONB NOT NULL,
    status job_status NOT NULL,
    schedule_time TIMESTAMP NOT NULL,
    priority INTEGER NOT NULL,
    worker_id TEXT,
    start_time TIMESTAMP,
    finish_time TIMESTAMP,
    cpu_usage SMALLINT,
    attempt INTEGER,
    error_message TEXT,
    error_stack_trace TEXT,
    CONSTRAINT pk_job PRIMARY KEY (id),
    CONSTRAINT check_task_length CHECK (char_length(task) <= 256),
    CONSTRAINT check_active_uid_length CHECK (char_length(active_uid) <= 512)
);

CREATE UNIQUE INDEX uk_job_active_uid ON job (active_uid) WHERE active_uid IS NOT NULL;
CREATE INDEX i_job_task_status_priority_schedule_time ON job (task, status, priority DESC, schedule_time);

CREATE TABLE task_stats
(
    task TEXT NOT NULL,
    stats JSONB NOT NULL,
    CONSTRAINT pk_task_stats PRIMARY KEY (task),
    CONSTRAINT check_task_length CHECK (char_length(task) <= 256)
);

CREATE TABLE sending_mail
(
    id                BIGSERIAL NOT NULL,
    job_id            UUID NOT NULL,
    parameters        JSONB NOT NULL,
    creation_ts       TIMESTAMP NOT NULL,
    CONSTRAINT pk_sending_mail_id PRIMARY KEY (id)
);

CREATE INDEX i_sending_mail_job_id ON sending_mail (job_id);

CREATE TABLE cron_job (
    task TEXT NOT NULL,
    create_time TIMESTAMP NOT NULL,
    status job_status NOT NULL,
    worker_id TEXT,
    start_time TIMESTAMP,
    finish_time TIMESTAMP,
    cpu_usage SMALLINT,
    error_message TEXT,
    error_stack_trace TEXT,
    execution_info JSONB,
    CONSTRAINT pk_cron_job PRIMARY KEY (task, create_time),
    CONSTRAINT check_task_length CHECK (char_length(task) <= 256)
);

CREATE TABLE cron_task (
    task TEXT NOT NULL,
    data JSONB NOT NULL,
    CONSTRAINT pk_cron_task PRIMARY KEY (task),
    CONSTRAINT check_task_length CHECK (char_length(task) <= 256)
);

CREATE TABLE cron_task_stats (
    task TEXT NOT NULL,
    stats JSONB NOT NULL,
    CONSTRAINT pk_cron_task_stats PRIMARY KEY (task),
    CONSTRAINT check_task_length CHECK (char_length(task) <= 256)
);

CREATE TABLE mailer_event (
    uid BIGINT NOT NULL,
    external_id TEXT NOT NULL,
    instance_id BIGINT NOT NULL,
    start_ts TIMESTAMP NOT NULL,
    end_ts TIMESTAMP NOT NULL,
    is_all_day BOOLEAN NOT NULL,
    repetition JSONB,
    recurrence_id TIMESTAMP,
    timezone_id TEXT NOT NULL,
    name TEXT NOT NULL,
    description TEXT,
    location TEXT,
    url TEXT,
    organizer JSONB,
    attendees JSONB NOT NULL,
    sequence INTEGER NOT NULL,
    dtstamp TIMESTAMP NOT NULL,
    created TIMESTAMP NOT NULL,
    modified TIMESTAMP NOT NULL,
    CONSTRAINT pk_mailer_event PRIMARY KEY (uid, external_id, instance_id),
    CONSTRAINT check_mailer_event_external_id_length CHECK (char_length(external_id) <= 512)
);