CREATE SCHEMA IF NOT EXISTS cerberus;
CREATE EXTENSION IF NOT EXISTS pg_partman SCHEMA public;

CREATE TABLE cerberus.resource_type
(
    name        TEXT   NOT NULL PRIMARY KEY CHECK (length(name) <= 20),
    description TEXT,
    action_set  TEXT[] NOT NULL CHECK (cardinality(action_set) > 0) -- list of the actions available for the given resource type
);

-- built-in resource types

CREATE TYPE cerberus.layer_action AS ENUM (
    'VIEW_EVENT',
    'EDIT_EVENT',
    'CREATE_EVENT',
    'DELETE_EVENT',
    'EDIT',
    'SHARE',
    'DELETE'
    );

INSERT INTO cerberus.resource_type (name, description, action_set)
VALUES ('layer', 'user layer', (SELECT enum_range(NULL::cerberus.layer_action)));

-- end of built-in resource types

CREATE TABLE cerberus.roles
(
    id          BIGSERIAL PRIMARY KEY,
    name        TEXT    NOT NULL UNIQUE,
    active      BOOLEAN NOT NULL,
    description TEXT
);

CREATE TABLE cerberus.groups
(
    id        BIGSERIAL NOT NULL,
    type      TEXT      NOT NULL DEFAULT 'internal',
    unique_id BIGSERIAL NOT NULL UNIQUE,
    name      TEXT      NOT NULL,
    active    BOOLEAN   NOT NULL,
    info      JSONB,
    roles     BIGINT[]  NOT NULL DEFAULT '{}', -- FK: cerberus.roles.id

    PRIMARY KEY (type, id)
);

CREATE INDEX idx_groups_type_name
    ON cerberus.groups (type, name);

CREATE TYPE cerberus.user_type AS ENUM (
    'BASIC',
    'YT'
    );

CREATE TABLE cerberus.users
(
    uid       BIGINT PRIMARY KEY,
    type      cerberus.user_type NOT NULL DEFAULT 'BASIC',
    login     TEXT     NOT NULL,
    superuser BOOLEAN  NOT NULL DEFAULT FALSE,
    info      JSONB,
    groups    BIGINT[] NOT NULL DEFAULT '{}', -- FK: cerberus.groups.unique_id
    roles     BIGINT[] NOT NULL DEFAULT '{}'  -- FK: cerberus.roles.id
);

CREATE TABLE cerberus.locations
(
    id   BIGSERIAL NOT NULL,
    type TEXT      NOT NULL,
    name TEXT      NOT NULL,
    info jsonb,

    PRIMARY KEY (type, id)
);

CREATE TABLE cerberus.resource
(
    id            BIGSERIAL NOT NULL,
    type          TEXT      NOT NULL REFERENCES cerberus.resource_type (name) ON DELETE RESTRICT,
    name          TEXT      NOT NULL,
    active        BOOLEAN   NOT NULL,
    location_id   BIGINT,
    location_type TEXT,
    info          JSONB,

    PRIMARY KEY (type, id),
    FOREIGN KEY (location_type, location_id) REFERENCES cerberus.locations (type, id) ON DELETE RESTRICT
);

CREATE TABLE cerberus.user_grants
(
    id            BIGSERIAL PRIMARY KEY,
    owner_id      BIGINT REFERENCES cerberus.users (uid) ON DELETE RESTRICT,
    resource_type TEXT   NOT NULL REFERENCES cerberus.resource_type (name) ON DELETE RESTRICT,
    resource_id   BIGINT,
    actions       TEXT[] NOT NULL CHECK (cardinality(actions) > 0),

    FOREIGN KEY (resource_type, resource_id) REFERENCES cerberus.resource (type, id) ON DELETE RESTRICT
);

CREATE UNIQUE INDEX idx_user_grants_owner_id_resource_type_resource_id
    ON cerberus.user_grants (owner_id, resource_type, resource_id);

CREATE TABLE cerberus.group_grants
(
    id            BIGSERIAL PRIMARY KEY,
    owner_id      BIGINT NOT NULL REFERENCES cerberus.groups (unique_id) ON DELETE RESTRICT,
    resource_type TEXT   NOT NULL REFERENCES cerberus.resource_type (name) ON DELETE RESTRICT,
    resource_id   BIGINT,
    actions       TEXT[] NOT NULL CHECK (cardinality(actions) > 0), -- granted actions

    FOREIGN KEY (resource_type, resource_id) REFERENCES cerberus.resource (type, id) ON DELETE RESTRICT
);

CREATE UNIQUE INDEX idx_group_grants_owner_id_resource_type_resource_id
    ON cerberus.group_grants (owner_id, resource_type, resource_id);

CREATE TABLE cerberus.role_grants
(
    id            BIGSERIAL PRIMARY KEY,
    owner_id      BIGINT NOT NULL REFERENCES cerberus.roles (id) ON DELETE RESTRICT,
    resource_type TEXT   NOT NULL REFERENCES cerberus.resource_type (name) ON DELETE RESTRICT,
    resource_id   BIGINT,
    actions       TEXT[] NOT NULL CHECK (cardinality(actions) > 0), -- granted actions

    FOREIGN KEY (resource_type, resource_id) REFERENCES cerberus.resource (type, id) ON DELETE RESTRICT
);

CREATE UNIQUE INDEX idx_role_grants_owner_id_resource_type_resource_id
    ON cerberus.role_grants (owner_id, resource_type, resource_id);

CREATE TYPE cerberus.change_action AS ENUM (
    'ADD',
    'UPDATE',
    'DELETE'
    );

CREATE TYPE cerberus.change_subject AS ENUM (
    'GROUP',
    'USER',
    'RESOURCE',
    'RESOURCE_TYPE',
    'ROLE',
    'USER_GRANT',
    'GROUP_GRANT',
    'ROLE_GRANT',
    'LOCATION'
    );

CREATE SEQUENCE cerberus.change_log_cid_seq;

CREATE TABLE cerberus.change_log
(
    cid                BIGINT                  NOT NULL DEFAULT NEXTVAL('cerberus.change_log_cid_seq'),
    action             cerberus.change_action  NOT NULL,
    subject_type       cerberus.change_subject NOT NULL,
    subject            JSONB                   NOT NULL,
    change_time        TIMESTAMPTZ             NOT NULL DEFAULT current_timestamp,
    actor_uid          BIGINT,
    actor_service_name TEXT                    NOT NULL,
    request_id         TEXT                    NOT NULL,
    change             JSONB                   NOT NULL
) PARTITION BY RANGE (change_time);

CREATE TABLE cerberus.change_log_template
(
    LIKE cerberus.change_log INCLUDING DEFAULTS
);

ALTER TABLE cerberus.change_log_template
    ADD CONSTRAINT pk_change_log_template PRIMARY KEY (cid);

SELECT create_parent(
               p_parent_table := 'cerberus.change_log',
               p_template_table := 'cerberus.change_log_template',
               p_control := 'change_time',
               p_type := 'native',
               p_interval := 'daily',
               p_premake := 5,
               p_jobmon := false
           );

UPDATE part_config
SET retention            = '90 days', -- TODO: fix right after YT backup will be configured
    retention_keep_table = false,
    retention_keep_index = false
WHERE parent_table = 'cerberus.change_log';

CREATE TYPE cerberus.task_status AS ENUM (
    'NEW',
    'STARTED',
    'FAILED',
    'SUCCESS',
    'TIMEOUT'
    );

CREATE TABLE cerberus.tasks
(
    idempotency_key UUID PRIMARY KEY,
    type            TEXT                 NOT NULL,
    schedule        TIMESTAMPTZ          NOT NULL,
    created         TIMESTAMPTZ          NOT NULL DEFAULT current_timestamp,
    started         TIMESTAMPTZ                   DEFAULT NULL,
    timeout         INTERVAL             NOT NULL,
    status          cerberus.task_status NOT NULL DEFAULT 'NEW',
    request_id      TEXT                 NOT NULL,
    initiator_uid   BIGINT,
    host            TEXT,
    context         jsonb,

    CONSTRAINT task_status_is_valid CHECK (status = 'NEW' OR status = 'STARTED'),
    CONSTRAINT task_created_before_schedule CHECK (created <= schedule),
    CONSTRAINT task_created_before_started CHECK (started IS NULL OR (created <= started)),
    CONSTRAINT task_scheduled_before_started CHECK (started IS NULL OR (schedule <= started)),
    CONSTRAINT task_host_is_present_if_task_started CHECK ((started IS NOT NULL AND host IS NOT NULL) OR
                                                           (started IS NULL AND host IS NULL))
);

CREATE INDEX idx_tasks_status_schedule
    ON cerberus.tasks (status, schedule);

CREATE INDEX idx_task_type_status
    ON cerberus.tasks (type, status);

CREATE TABLE cerberus.recent_tasks
(
    idempotency_key UUID                 NOT NULL,
    type            TEXT                 NOT NULL,
    created         TIMESTAMPTZ          NOT NULL,
    started         TIMESTAMPTZ          NOT NULL,
    finished        TIMESTAMPTZ          NOT NULL DEFAULT current_timestamp,
    status          cerberus.task_status NOT NULL,
    request_id      TEXT                 NOT NULL,
    initiator_uid   BIGINT,
    error           TEXT                          DEFAULT NULL,
    host            TEXT                 NOT NULL,
    context         jsonb
) PARTITION BY RANGE (finished);

CREATE TABLE cerberus.recent_tasks_template
(
    LIKE cerberus.recent_tasks INCLUDING DEFAULTS
);

ALTER TABLE cerberus.recent_tasks_template
    ADD CONSTRAINT recent_task_status_is_valid CHECK (
        status = 'FAILED' OR status = 'SUCCESS' OR status = 'TIMEOUT'
        ),
    ADD CONSTRAINT recent_task_started_is_present CHECK (started IS NOT NULL),
    ADD CONSTRAINT recent_task_finished_is_present CHECK (finished IS NOT NULL),
    ADD CONSTRAINT recent_task_started_before_finished CHECK (started < finished),
    ADD CONSTRAINT recent_task_failed_status_with_error CHECK (
            ((status = 'FAILED' OR status = 'TIMEOUT') AND error IS NOT NULL) OR
            (status = 'SUCCESS' AND error IS NULL)
        );

SELECT create_parent(
               p_parent_table := 'cerberus.recent_tasks',
               p_template_table := 'cerberus.recent_tasks_template',
               p_control := 'finished',
               p_type := 'native',
               p_interval := 'daily',
               p_premake := 7,
               p_jobmon := false
           );

UPDATE part_config
SET retention            = '7 days',
    retention_keep_table = false,
    retention_keep_index = false
WHERE parent_table = 'cerberus.recent_tasks';
