CREATE OR REPLACE PROCEDURE cerberus.repartition(p_schema_name TEXT, p_table_name TEXT, p_template_table_name TEXT,
                                                 p_control TEXT, p_premake INTEGER, p_retention TEXT,
                                                 p_template_primary_key_constraint_name TEXT DEFAULT NULL) AS $$
DECLARE
    full_name TEXT = p_schema_name || '.' || p_table_name;
    tmp_name TEXT = p_table_name || '_tmp';
    tmp_full_name TEXT = p_schema_name || '.' || tmp_name;
    tmp_template_full_name TEXT = tmp_full_name || '_template';
    final_template_name TEXT = p_table_name || '_template';
    start_partition TEXT;
    partition RECORD;
    create_parent_result BOOLEAN;
BEGIN
    EXECUTE 'LOCK TABLE ' || full_name;

    -- drop primary key from source table template if it doesn't include partition key
    IF p_template_primary_key_constraint_name IS NOT NULL THEN
        EXECUTE format('ALTER TABLE %1$s DROP CONSTRAINT %2$s',
            p_schema_name || '.' || p_template_table_name,
            p_template_primary_key_constraint_name);
    END IF;

    -- select first partition ts
    EXECUTE format('SELECT min(%1$s)::TEXT FROM %2$s', p_control, full_name)
        INTO start_partition;

    -- create tmp table
    EXECUTE format('CREATE TABLE %1$s (LIKE %2$s '
                     'INCLUDING COMMENTS '
                     'INCLUDING CONSTRAINTS '
                     'INCLUDING DEFAULTS '
                     'INCLUDING IDENTITY '
                     'INCLUDING INDEXES '
                   ') PARTITION BY RANGE (%3$s)',
        tmp_full_name, p_schema_name || '.' || p_template_table_name, p_control);

    -- create empty template table
    EXECUTE format('CREATE TABLE %1$s ()', tmp_template_full_name);

    -- create tmp table partitions
    SELECT create_parent(
               p_parent_table := tmp_full_name,
               p_template_table := tmp_template_full_name,
               p_control := p_control,
               p_type := 'native',
               p_interval := 'daily',
               p_premake := repartition.p_premake,
               p_start_partition := start_partition,
               p_jobmon := false,
               p_debug := true
           )
    INTO create_parent_result;

    IF NOT create_parent_result THEN
        RAISE EXCEPTION 'create_parent failed';
    END IF;

    -- configure tmp table retention
    UPDATE part_config
        SET retention                = p_retention,
            retention_keep_table     = false,
            retention_keep_index     = false,
            infinite_time_partitions = true
    WHERE parent_table = tmp_full_name;

    -- copy data into tmp table
    EXECUTE format('INSERT INTO %1$s '
                   'SELECT * '
                   'FROM %2$s',
        tmp_full_name, full_name);

    -- drop template table
    IF p_template_table_name IS NOT NULL THEN
        EXECUTE 'DROP TABLE ' || p_schema_name || '.' || p_template_table_name;
    END IF;

    -- drop source table
    EXECUTE 'DROP TABLE ' || full_name;
    -- drop source table default partition
    EXECUTE 'DROP TABLE IF EXISTS ' || full_name || '_default';

    -- remove outdated partman configuration
    DELETE FROM public.part_config
    WHERE parent_table = full_name;

    -- rename tmp table to source table
    EXECUTE 'ALTER TABLE ' || tmp_full_name || ' RENAME TO ' || p_table_name;
    EXECUTE 'ALTER TABLE ' || tmp_full_name || '_default RENAME TO ' || p_table_name || '_default';
    EXECUTE 'ALTER TABLE ' || tmp_template_full_name || ' RENAME TO ' || final_template_name;

    -- rename partitions
    FOR partition IN
        SELECT table_name
        FROM information_schema.tables
        WHERE table_schema = p_schema_name AND table_type = 'BASE TABLE'
          AND starts_with(table_name, tmp_name || '_p')
    LOOP
        EXECUTE format('ALTER TABLE %1$s.%2$s RENAME TO %3$s',
            p_schema_name,
            partition.table_name,
            p_table_name || '_p' || substr(partition.table_name, length(tmp_name || '_p') + 1));
    END LOOP;

    -- rename tmp table configuration in pg_partman config
    UPDATE public.part_config
        SET parent_table = full_name,
            template_table = p_schema_name || '.' || final_template_name
    WHERE parent_table = tmp_full_name;

    PERFORM run_maintenance(
        p_parent_table := full_name,
        p_analyze := true,
        p_jobmon := false,
        p_debug := true
        );
END
$$  LANGUAGE plpgsql;

CALL cerberus.repartition(
    p_schema_name := 'cerberus',
    p_table_name := 'recent_tasks',
    p_template_table_name := 'recent_tasks_template',
    p_control := 'finished',
    p_premake := 5,
    p_retention := '7 days'
    );

CALL cerberus.repartition(
    p_schema_name := 'cerberus',
    p_table_name := 'change_log',
    p_template_table_name := 'change_log_template',
    p_template_primary_key_constraint_name := 'pk_change_log_template',
    p_control := 'change_time',
    p_premake := 5,
    p_retention := '1 year'
    );

DROP PROCEDURE cerberus.repartition;
