CREATE SCHEMA IF NOT EXISTS transfer;

CREATE TABLE transfer.info (
    uid bigint,
    suid bigint,
    from_db text,
    to_db text,
    transfer_start timestamp with time zone,
    transfer_end timestamp with time zone,
    script text,
    script_revision text,
    transfer_host text,
    CONSTRAINT pk_transfer_info PRIMARY KEY (uid, transfer_start, transfer_end)
);


CREATE SEQUENCE transfer.huskies_s;

CREATE TYPE transfer.husky_status AS ENUM (
    'running',
    'sleeping',
    'dead'
);

CREATE TABLE transfer.huskies (
    husky_id     integer NOT NULL,
    host         text,
    status       transfer.husky_status    NOT NULL DEFAULT 'sleeping',
    heartbeated  timestamp with time zone NOT NULL DEFAULT current_timestamp,
    usercount    integer NOT NULL DEFAULT 0,

    CONSTRAINT pk_husies PRIMARY KEY (husky_id)
);


CREATE TYPE transfer.task AS ENUM (
    'transfer',
    'reset_passport_db',
    'fix',
    'call_sql',
    'clone_user'
);

CREATE TYPE transfer.task_status AS ENUM (
    'do_not_process',
    'userinfo_unfilled',
    'pending',
    'in_progress',
    'complete',
    'error'
);

CREATE TYPE transfer.error AS ENUM (
    'transfer_invalid_from_db',
    'no_such_user',
    'unknown',
    'not_supported',
    'wrong_args'
);

CREATE TABLE transfer.users_in_dogsleds (
    transfer_id SERIAL PRIMARY KEY,
    uid         bigint,
    suid        bigint,
    login       text,

    priority    smallint NOT NULL DEFAULT 0,
    husky_id    integer,

    tries       smallint NOT NULL DEFAULT 0,
    task        transfer.task            NOT NULL DEFAULT 'transfer',
    task_args   jsonb,
    status      transfer.task_status     NOT NULL DEFAULT 'pending',
    error_type  transfer.error,
    last_update timestamp with time zone NOT NULL DEFAULT current_timestamp,
    try_notices text[],

    CONSTRAINT check_tries CHECK (tries >= 0),
    CONSTRAINT check_status CHECK (status = 'pending' OR husky_id IS NOT NULL),
    CONSTRAINT check_userinfo CHECK (
            uid IS NOT NULL
         OR suid IS NOT NULL
         OR login IS NOT NULL
    ),
    CONSTRAINT check_uid_and_status CHECK (
            uid IS NOT NULL
         OR (
            status = 'userinfo_unfilled'
            OR (status = 'error' AND error_type IN ('no_such_user', 'unknown')))
    ),

    CONSTRAINT fk_users_in_dogsleds_and_huskies FOREIGN KEY (husky_id)
        REFERENCES transfer.huskies ON DELETE RESTRICT
);

COMMENT ON COLUMN transfer.users_in_dogsleds.priority
    IS 'lowest priority = first to transfer';

CREATE UNIQUE INDEX i_users_in_dogsleds_by_uid_not_null_not_processed
    ON transfer.users_in_dogsleds (uid, task)
 WHERE uid IS NOT NULL
   AND status = 'pending';

COMMENT ON INDEX transfer.i_users_in_dogsleds_by_uid_not_null_not_processed
    IS 'Denies planning or executing two same tasks on same user concurrently, '
       'allows lookup of currently planned/executed tasks on user';

CREATE INDEX i_users_in_dogsleds_by_husky_id_status_prioretized
    ON transfer.users_in_dogsleds (husky_id, status, priority, last_update);

COMMENT ON INDEX transfer.i_users_in_dogsleds_by_husky_id_status_prioretized
    IS 'Works as prioritized queue for each husky_id';

CREATE INDEX i_users_in_dogsleds_by_uid
    ON transfer.users_in_dogsleds (uid);


CREATE TABLE transfer.clone_user_audit (
    source_uid   bigint NOT NULL,
    dest_uid     bigint NOT NULL UNIQUE,
    transfer_id  bigint NOT NULL UNIQUE,
    cloned_ts    timestamp with time zone NOT NULL DEFAULT current_timestamp,
    request_info jsonb  NOT NULL,
    complete     bool   NOT NULL DEFAULT FALSE,
    complete_at  timestamp with time zone,

    CONSTRAINT pk_clone_user_audit PRIMARY KEY (
        source_uid, dest_uid
    )
);
