DROP SCHEMA IF EXISTS transfer CASCADE;

DELETE FROM public.part_config
WHERE parent_table = 'transfer.processed_tasks';

CREATE SCHEMA transfer;

CREATE TABLE transfer.info (
    uid bigint,
    from_db text,
    to_db text,
    transfer_start timestamp with time zone,
    transfer_end timestamp with time zone,
    script text,
    script_revision text,
    transfer_host text,
    CONSTRAINT pk_transfer_info PRIMARY KEY (uid, transfer_start, transfer_end)
);

CREATE TYPE transfer.task AS ENUM (
    'transfer',
    'clone_user',
    'apply_data_migrations',
    'delete_mail_user',
    'delete_shards_user',
    'put_letter',
    'copy_abook',
    'move_messages_to_tabs',
    'cut_firstlines',
    'unlabel_messages',
    'onetime_task'
);

CREATE TYPE transfer.task_status AS ENUM (
    'do_not_process',
    'pending',
    'in_progress',
    'complete',
    'error'
);

CREATE TYPE transfer.error AS ENUM (
    'transfer_invalid_from_db',
    'no_such_user',
    'unknown',
    'not_supported',
    'wrong_args',
    'failed_write_data',
    'failed_get_data',
    'closed_dest_shard',
    'deferred_due_to_message_count_limit'
);

CREATE TYPE transfer.user_type AS ENUM (
    'passport_user',
    'passport_domain',
    'connect_organization'
);


CREATE TABLE transfer.task_group (
    id            serial PRIMARY KEY,
    name          text NOT NULL,
    husky_cluster text NOT NULL
);

CREATE INDEX i_task_group_by_husky_cluster
    ON transfer.task_group (husky_cluster);


CREATE TABLE transfer.users_in_dogsleds (
    transfer_id bigserial PRIMARY KEY,
    uid         bigint NOT NULL,

    priority    smallint NOT NULL DEFAULT 0,

    tries       smallint NOT NULL DEFAULT 0,
    task        transfer.task            NOT NULL DEFAULT 'transfer',
    task_args   jsonb,
    status      transfer.task_status     NOT NULL DEFAULT 'pending',
    error_type  transfer.error,
    planned     timestamp with time zone NOT NULL DEFAULT current_timestamp,
    last_update timestamp with time zone NOT NULL DEFAULT current_timestamp,
    try_notices text[],

    task_output jsonb,
    user_type transfer.user_type,
    shard_id    integer NOT NULL,
    task_group_id integer REFERENCES transfer.task_group (id),

    CONSTRAINT check_tries CHECK (tries >= 0)
);

COMMENT ON COLUMN transfer.users_in_dogsleds.priority
    IS 'lowest priority = first to transfer';

CREATE UNIQUE INDEX i_users_in_dogsleds_by_uid_not_processed
    ON transfer.users_in_dogsleds (uid, task)
 WHERE status NOT IN (
    'complete'::transfer.task_status,
    'error'::transfer.task_status
);

CREATE INDEX i_users_in_dogsleds_for_husky_worker_prioritized
    ON transfer.users_in_dogsleds (shard_id, status, priority, planned);

COMMENT ON INDEX transfer.i_users_in_dogsleds_for_husky_worker_prioritized
    IS 'Works as prioritized queue for new huskies';

CREATE INDEX i_users_in_dogsleds_by_uid
    ON transfer.users_in_dogsleds (uid);

CREATE INDEX i_users_in_dogsleds_for_clone
    ON transfer.users_in_dogsleds
 USING btree (status)
 WHERE (task = 'clone_user'::transfer.task);


CREATE TABLE transfer.clone_user_audit (
    source_uid   bigint NOT NULL,
    dest_uid     bigint NOT NULL UNIQUE,
    transfer_id  bigint NOT NULL UNIQUE,
    created      timestamp with time zone NOT NULL DEFAULT current_timestamp,
    request_info jsonb  NOT NULL,
    complete     bool   NOT NULL DEFAULT FALSE,
    complete_at  timestamp with time zone,

    CONSTRAINT pk_clone_user_audit PRIMARY KEY (
        source_uid, dest_uid
    )
);

CREATE TABLE transfer.global_task_args (
    args_id text NOT NULL,
    task_args jsonb,

    CONSTRAINT pk_global_task_args PRIMARY KEY (args_id)
);


CREATE TABLE transfer.stoplist (
    id serial,
    uid bigint NOT NULL,
    reason text DEFAULT ''::text NOT NULL,

    CONSTRAINT stoplist_pkey PRIMARY KEY (id)
);


CREATE TABLE transfer.clone_st_ids_with_mime_copies (
    uid bigint NOT NULL,
    st_id text NOT NULL,
    new_st_id text NOT NULL,
    mime_xml bytea,

    CONSTRAINT pk_clone_st_ids_with_mime_copies PRIMARY KEY (uid, st_id)
);


CREATE TABLE transfer.processed_tasks (
    transfer_id bigint,
    uid         bigint,
    priority    smallint,
    tries       smallint,
    task        transfer.task,
    task_args   jsonb,
    status      transfer.task_status,
    error_type  transfer.error,
    planned     timestamp with time zone,
    last_update timestamp with time zone,
    try_notices text[],
    task_output jsonb,
    shard_id    integer
) PARTITION BY RANGE (last_update);

CREATE TABLE transfer.processed_tasks_templ (
    LIKE transfer.processed_tasks INCLUDING DEFAULTS
);

ALTER TABLE transfer.processed_tasks_templ
    ADD CONSTRAINT pk_processed_tasks_templ PRIMARY KEY (transfer_id);

CREATE INDEX i_processed_tasks_by_uid_templ
    ON transfer.processed_tasks_templ (uid);

CREATE INDEX i_processed_tasks_by_task_and_shard_id_templ
    ON transfer.processed_tasks_templ (task, shard_id, task_args);

SELECT create_parent(
    p_parent_table := 'transfer.processed_tasks',
    p_template_table := 'transfer.processed_tasks_templ',
    p_control := 'last_update',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 5,
    p_jobmon := false
);
UPDATE part_config
   SET retention = '60 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'transfer.processed_tasks';
