CREATE SCHEMA IF NOT EXISTS shiva;

CREATE TYPE shiva.shard_types AS ENUM (
    'canary',
    'canary2',
    'qa',
    'general'
);

CREATE TABLE shiva.shards (
    shard_id            integer NOT NULL,
    cluster_id          TEXT NOT NULL,
    shard_type          shiva.shard_types NOT NULL DEFAULT 'general',
    migration           integer NOT NULL,
    priority            integer NOT NULL DEFAULT 0,
    disk_size           bigint NOT NULL,
    shard_name          TEXT,
    load_type           TEXT NOT NULL DEFAULT 'general',
    can_transfer_to     boolean NOT NULL DEFAULT false,
    used_size           bigint NOT NULL DEFAULT 0,

    CONSTRAINT pk_shards PRIMARY KEY (shard_id)
);

CREATE UNIQUE INDEX uk_shards_cluster_id
    ON shiva.shards (cluster_id);

CREATE INDEX i_shards_shard_type ON shiva.shards (shard_type);


CREATE TYPE shiva.status AS ENUM (
    'pending',
    'in_progress',
    'complete',
    'error'
);

CREATE TABLE shiva.migrations (
    migration_id         SERIAL PRIMARY KEY,
    migration            integer NOT NULL,
    migration_args       jsonb,
    status               shiva.status NOT NULL DEFAULT 'in_progress'
);

CREATE UNIQUE INDEX i_migrations_by_status_not_processed
    ON shiva.migrations (status)
 WHERE status NOT IN ('complete', 'error');


CREATE TABLE shiva.migrations_info (
    migration_id         integer NOT NULL,
    shard_id             integer NOT NULL,
    prev_migration       integer NOT NULL,
    status               shiva.status NOT NULL DEFAULT 'pending',
    notice               TEXT,
    migrate_log          TEXT,

    CONSTRAINT pk_migrations_info PRIMARY KEY (migration_id, shard_id),

    CONSTRAINT fk_migrations_and_migrations_info
        FOREIGN KEY (migration_id) REFERENCES shiva.migrations ON DELETE CASCADE
);

CREATE TYPE shiva.task_types AS ENUM (
    'cleanup_doomed',
    'end_prepared_transaction',
    'folder_archivation',
    'init_pop3_folders',
    'pg_partman_maintenance',
    'purge_chained_log',
    'purge_deleted_box',
    'purge_storage',
    'purge_user',
    'space_balancer',
    'update_mailbox_size',
    'settings_export',
    'huskydb_pg_partman_maintenance',
    'close_for_load',
    'transfer_users',
    'purge_transferred_user',
    'purge_deleted_user',
    'purge_synced_deleted_box',
    'transfer_active_users',
    'pnl_estimation_export',
    'reactivate_users',
    'deactivate_users',
    'start_freezing_users',
    'notify_users',
    'freeze_users',
    'archive_users',
    'delete_archived_users',
    'queuedb_pg_partman_maintenance',
    'purge_backups',
    'purge_archives',
    'clean_archives',
    'callmebackdb_clean',
    'onetime_task'
);

CREATE TABLE shiva.shard_running_tasks (
    shard_id            integer NOT NULL,
    task                shiva.task_types NOT NULL,
    job_no              integer NOT NULL,
    started             timestamp with time zone NOT NULL DEFAULT current_timestamp,
    worker              TEXT NOT NULL,

    CONSTRAINT pk_running_tasks PRIMARY KEY (shard_id, task, job_no)
);

CREATE INDEX i_running_tasks_by_worker
    ON shiva.shard_running_tasks (worker);


CREATE TABLE shiva.shard_tasks_info (
    shard_id            integer NOT NULL,
    task                shiva.task_types NOT NULL,
    job_no              integer NOT NULL,
    started             timestamp with time zone NOT NULL,
    finished            timestamp with time zone NOT NULL DEFAULT current_timestamp,
    worker              TEXT NOT NULL,
    status              shiva.status NOT NULL,
    notice              TEXT
) PARTITION BY RANGE (finished);

CREATE TABLE shiva.shard_tasks_info_templ (
    LIKE shiva.shard_tasks_info INCLUDING DEFAULTS
);

ALTER TABLE shiva.shard_tasks_info_templ
    ADD CONSTRAINT pk_shard_tasks_info_templ PRIMARY KEY (shard_id, task, job_no, started, finished);

CREATE INDEX i_shard_tasks_info_by_status_task_worker_templ
    ON shiva.shard_tasks_info_templ (status, task, worker);

SELECT create_parent(
    p_parent_table := 'shiva.shard_tasks_info',
    p_template_table := 'shiva.shard_tasks_info_templ',
    p_control := 'finished',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 5,
    p_jobmon := false
);
UPDATE part_config
   SET retention = '7 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'shiva.shard_tasks_info';
