DROP SCHEMA IF EXISTS backup CASCADE;
CREATE SCHEMA backup;

CREATE TYPE backup.backup_state AS ENUM (
    'in_progress',
    'complete',
    'inactive',
    'error'
);

CREATE TYPE backup.restore_state AS ENUM (
    'in_progress',
    'complete',
    'error'
);

CREATE TABLE backup.backups (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,

    version             integer NOT NULL DEFAULT 1,
    message_count       integer NOT NULL DEFAULT 0,
    state               backup.backup_state NOT NULL DEFAULT 'in_progress',
    revision            bigint,
    created             timestamptz NOT NULL DEFAULT current_timestamp,
    updated             timestamptz NOT NULL DEFAULT current_timestamp,
    notice              text,

    CONSTRAINT pk_backups PRIMARY KEY (uid, backup_id),
    CONSTRAINT fk_backups_mail_users FOREIGN KEY (uid) REFERENCES mail.users ON DELETE RESTRICT,

    CONSTRAINT check_message_count CHECK (
        message_count >= 0
    )
);

COMMENT ON COLUMN backup.backups.message_count
    IS 'the number of backuped e-mails';

CREATE INDEX i_backups_state_updated
    ON backup.backups (state, updated);

CREATE INDEX i_backups_uid_state
    ON backup.backups (uid, state);

CREATE UNIQUE INDEX uk_backups_uid_state_in_progress
    ON backup.backups (uid)
 WHERE state = 'in_progress';

CREATE UNIQUE INDEX uk_backups_uid_state_complete
    ON backup.backups (uid)
 WHERE state = 'complete';

CREATE TYPE backup.restore_method AS ENUM (
    'full_hierarchy',
    'restored_folder'
);

CREATE TYPE backup.fid_to_fid AS (
    original integer,
    renewed  integer
);

CREATE TYPE backup.mid_to_mid AS (
    original bigint,
    renewed  bigint
);

CREATE TABLE backup.restores (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,
    created             timestamptz NOT NULL DEFAULT current_timestamp,
    updated             timestamptz NOT NULL DEFAULT current_timestamp,
    to_restore_count    integer NOT NULL DEFAULT 0,
    restored_count      integer NOT NULL DEFAULT 0,
    state               backup.restore_state NOT NULL DEFAULT 'in_progress',
    method              backup.restore_method NOT NULL,
    notice              text,
    fids_mapping        backup.fid_to_fid[] NOT NULL DEFAULT '{}',

    CONSTRAINT pk_restores PRIMARY KEY (uid, backup_id, created),
    CONSTRAINT fk_restores_backups FOREIGN KEY (uid, backup_id) REFERENCES backup.backups ON DELETE RESTRICT,

    CONSTRAINT check_to_restore_count CHECK (
        to_restore_count >= 0
    ),

    CONSTRAINT check_restored_count CHECK (
        restored_count >= 0
    )
);

COMMENT ON COLUMN backup.restores.to_restore_count
    IS 'the number of emails that need to be restored';

COMMENT ON COLUMN backup.restores.restored_count
    IS 'the number of emails already restored';

CREATE INDEX i_restores_uid_state
    ON backup.restores (uid, state);

CREATE UNIQUE INDEX uk_restores_uid_state
    ON backup.restores (uid)
 WHERE state = 'in_progress';

CREATE TABLE backup.folders (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,

    fid                 integer NOT NULL,
    type                mail.folder_types NOT NULL,
    name                text NOT NULL,
    parent_fid          integer,
    is_backuped         boolean NOT NULL DEFAULT false,

    CONSTRAINT pk_folders PRIMARY KEY (uid, backup_id, fid),
    CONSTRAINT fk_folders_backups FOREIGN KEY (uid, backup_id) REFERENCES backup.backups ON DELETE RESTRICT
);

COMMENT ON TABLE backup.folders
     IS 'copy of the mail.folders at the time of the backup';


CREATE TABLE backup.box (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,
    mid                 bigint NOT NULL,

    st_id               text NOT NULL,
    fid                 integer NOT NULL,
    tab                 mail.tab_types,
    received_date       timestamptz NOT NULL,
    attributes          mail.message_attributes[] DEFAULT '{}'
        CHECK (array_ndims(attributes) = 1),

    CONSTRAINT pk_box PRIMARY KEY (uid, backup_id, mid),
    CONSTRAINT fk_box_folders FOREIGN KEY (uid, backup_id, fid) REFERENCES backup.folders ON DELETE RESTRICT
);

CREATE INDEX i_box_uid_backup_id_received_date
    ON backup.box (uid, backup_id, received_date);

CREATE INDEX i_box_uid_backup_id_fid
    ON backup.box (uid, backup_id, fid);

CREATE INDEX if_box_hashtext_st_id
    ON backup.box (hashtext(st_id));


CREATE TABLE backup.folders_to_backup (
    uid                 bigint NOT NULL,
    fid                 integer NOT NULL,

    CONSTRAINT pk_folders_to_backup PRIMARY KEY (uid, fid),
    CONSTRAINT fk_folders_mail_folders FOREIGN KEY (uid, fid) REFERENCES mail.folders ON DELETE CASCADE
);

COMMENT ON TABLE backup.folders_to_backup
     IS 'for specifying the folders to be backed up';


CREATE TABLE backup.tabs_to_backup (
    uid                 bigint NOT NULL,
    tab                 mail.tab_types NOT NULL,

    CONSTRAINT pk_tabs_to_backup PRIMARY KEY (uid, tab),
    CONSTRAINT fk_tabs_mail_tabs FOREIGN KEY (uid, tab) REFERENCES mail.tabs ON DELETE CASCADE
);

COMMENT ON TABLE backup.tabs_to_backup
     IS 'for specifying the tabs to be backed up';


CREATE TYPE backup.settings AS (
    fids    integer[],
    tabs    mail.tab_types[]
);
