CREATE OR REPLACE FUNCTION code.create_backup(
    i_uid             code.uid,
    i_backup_id       integer,
    i_max_messages    integer,
    i_use_tabs        boolean,
    i_request_info    code.request_info DEFAULT NULL
) RETURNS text AS $$
DECLARE
    current_revision    bigint;
    max_backup_id       integer;
    messages_in_folders integer;
    messages_in_tabs    integer;
    folders_count       integer;
    tabs_count          integer;
    running_restore     boolean;
    inbox_fid           integer := code.get_folder_fid(i_uid, 'inbox');
BEGIN
    SELECT next_revision, next_backup_id
      FROM impl.acquire_current_serials(i_uid)
      INTO current_revision, max_backup_id;

    IF i_backup_id >= max_backup_id THEN
        RETURN 'backup_id_not_reserved';
    END IF;

    SELECT count(*)
      FROM backup.folders_to_backup
     WHERE uid = i_uid
      INTO folders_count;

    SELECT count(*)
      FROM backup.tabs_to_backup
     WHERE uid = i_uid
      INTO tabs_count;

    IF COALESCE(folders_count, 0) = 0 AND COALESCE(tabs_count, 0) = 0 THEN
        RETURN 'empty_folder_list';
    END IF;

    SELECT sum(mf.message_count)
      FROM mail.folders AS mf
      JOIN backup.folders_to_backup AS ftb
        ON mf.uid = ftb.uid AND mf.fid = ftb.fid
     WHERE mf.uid = i_uid
       AND (NOT i_use_tabs OR ftb.fid != inbox_fid)
      INTO messages_in_folders;

    SELECT sum(mt.message_count)
      FROM mail.tabs AS mt
      JOIN backup.tabs_to_backup AS ttb
        ON mt.uid = ttb.uid AND mt.tab = ttb.tab
     WHERE mt.uid = i_uid
      INTO messages_in_tabs;

    IF (COALESCE(messages_in_folders, 0) + COALESCE(messages_in_tabs, 0)) >= i_max_messages THEN
        RETURN 'too_many_messages';
    ELSIF COALESCE(messages_in_folders, 0) = 0 AND COALESCE(messages_in_tabs, 0) = 0 THEN
        RETURN 'nothing_to_backup';
    END IF;

    SELECT true
      FROM backup.restores
     WHERE uid = i_uid AND state = 'in_progress'
      INTO running_restore;

    IF running_restore THEN
        RETURN 'running_restore';
    END IF;

    INSERT INTO backup.backups (uid, backup_id, revision)
    VALUES (i_uid, i_backup_id, current_revision)
        ON CONFLICT DO NOTHING;

    IF NOT FOUND THEN
        RETURN 'unique_violation';
    END IF;

    RETURN NULL;
END;
$$ LANGUAGE plpgsql;
