CREATE OR REPLACE FUNCTION code.fill_backup(
    i_uid             code.uid,
    i_backup_id       integer,
    i_use_tabs        boolean,
    i_request_info    code.request_info DEFAULT NULL
) RETURNS text AS $$
DECLARE
    current_state backup.backup_state;
    messages      integer;
    inbox_fid     integer := code.get_folder_fid(i_uid, 'inbox');
BEGIN
    PERFORM code.acquire_current_revision(i_uid);

    -- Backup should exist and be in `in_progress` state for filling
    SELECT state
      FROM backup.backups
     WHERE uid = i_uid AND backup_id = i_backup_id
      INTO current_state;

    IF current_state IS NULL THEN
        RETURN 'backup_is_missing';
    ELSIF current_state != 'in_progress' THEN
        RETURN 'wrong_state';
    END IF;

    -- Dump all user folders to backup and set `is_backuped = true` only for folders in settings
    -- We need this for restoring full folders hierarchy
    INSERT INTO backup.folders (uid, backup_id, fid, type, name, parent_fid, is_backuped)
    SELECT i_uid, i_backup_id, mf.fid, mf.type, mf.name, mf.parent_fid, ftb.uid IS NOT NULL
      FROM mail.folders AS mf
      LEFT JOIN backup.folders_to_backup AS ftb
        ON mf.uid = ftb.uid AND mf.fid = ftb.fid
     WHERE mf.uid = i_uid;

    -- Dump messages from folders in settings
    -- The weird condition with i_use_tabs is for skipping inbox
    --   if tabs are `on` we should dump from tabs and not from inbox
    INSERT INTO backup.box (uid, backup_id, mid, st_id, fid, tab, received_date, attributes)
    SELECT i_uid, i_backup_id, mb.mid, mm.st_id, mb.fid, mb.tab, mb.received_date, mm.attributes
      FROM mail.box AS mb
      JOIN backup.folders_to_backup AS ftb
        ON mb.uid = ftb.uid AND mb.fid = ftb.fid
      JOIN mail.messages AS mm
        ON mb.uid = mm.uid AND mb.mid = mm.mid
     WHERE mb.uid = i_uid
       AND (NOT i_use_tabs OR ftb.fid != inbox_fid);

    -- Dump messages from tabs in settings
    -- We should do this only if tabs are `on`
    --   otherwise messages from tabs already in backup via inbox
    IF i_use_tabs THEN
        INSERT INTO backup.box (uid, backup_id, mid, st_id, fid, tab, received_date, attributes)
        SELECT i_uid, i_backup_id, mb.mid, mm.st_id, mb.fid, mb.tab, mb.received_date, mm.attributes
          FROM mail.box AS mb
          JOIN backup.tabs_to_backup AS ttb
            ON mb.uid = ttb.uid AND mb.tab = ttb.tab
          JOIN mail.messages AS mm
            ON mb.uid = mm.uid AND mb.mid = mm.mid
         WHERE mb.uid = i_uid;
    END IF;

    -- Count total messages in backup
     SELECT count(*)
       FROM backup.box
      WHERE uid = i_uid AND backup_id = i_backup_id
       INTO messages;

    IF messages = 0 THEN
        UPDATE backup.backups
           SET state = 'error',
               updated = current_timestamp
         WHERE uid = i_uid AND backup_id = i_backup_id;
        RETURN 'nothing_to_backup';
    END IF;

    -- Deactivate previous backup
     UPDATE backup.backups
        SET state = 'inactive',
            updated = current_timestamp
      WHERE uid = i_uid AND state = 'complete';

    -- Update current backup state and message_count
     UPDATE backup.backups
        SET state = 'complete',
            updated = current_timestamp,
            message_count = messages
      WHERE uid = i_uid AND backup_id = i_backup_id;

    RETURN NULL;
END;
$$ LANGUAGE plpgsql;
