CREATE OR REPLACE FUNCTION code.get_heap_bloat_info()
RETURNS TABLE (
    current_database name, 
    schemaname name, 
    tblname name, 
    real_size text, 
    extra_size text, 
    extra_ratio float, 
    fillfactor integer, 
    bloat_size text, 
    bloat_size_bytes bigint, 
    bloat_ratio float, 
    is_na boolean 
) AS $$
BEGIN
RETURN QUERY WITH s AS (
SELECT
        tbl.oid AS tblid, 
    ns.nspname AS schemaname, 
    tbl.relname AS tblname, 
    tbl.reltuples,
        tbl.relpages AS heappages, 
    COALESCE(toast.relpages, 0) AS toastpages,
        COALESCE(toast.reltuples, 0) AS toasttuples,
        COALESCE(substring(array_to_string(tbl.reloptions, ' ')
          FROM '%fillfactor=#"__#"%' FOR '#')::smallint, 100) AS fillfactor,
        current_setting('block_size')::numeric AS bs,
        CASE WHEN version()~'mingw32' OR version()~'64-bit|x86_64|ppc64|ia64|amd64' 
    THEN 8 
    ELSE 4 
    END AS ma,
        24 AS page_hdr,
        23 + CASE WHEN MAX(coalesce(null_frac,0)) > 0 
    THEN ( 7 + count(*) ) / 8 
    ELSE 0::int 
    END
          + CASE WHEN tbl.relhasoids 
    THEN 4 
    ELSE 0 
    END AS tpl_hdr_size,
        sum( (1-coalesce(s.null_frac, 0)) * coalesce(s.avg_width, 1024) ) AS tpl_data_size,
        bool_or(att.atttypid = 'pg_catalog.name'::regtype) AS is_na
FROM pg_attribute AS att
        JOIN pg_class AS tbl ON att.attrelid = tbl.oid
        JOIN pg_namespace AS ns ON ns.oid = tbl.relnamespace
        JOIN pg_stats AS s ON s.schemaname=ns.nspname
          AND s.tablename = tbl.relname AND s.inherited=false AND s.attname=att.attname
        LEFT JOIN pg_class AS toast ON tbl.reltoastrelid = toast.oid
WHERE att.attnum > 0 AND NOT att.attisdropped
        AND tbl.relkind = 'r'
GROUP BY 
    1,2,3,4,5,6,7,8,9,10, tbl.relhasoids
ORDER BY 
    5 DESC
), s2 AS (
SELECT
        ( 4 + tpl_hdr_size + tpl_data_size + (2*ma)
        - CASE WHEN tpl_hdr_size%ma = 0 
    THEN ma 
    ELSE tpl_hdr_size%ma 
    END
        - CASE WHEN ceil(tpl_data_size)::int%ma = 0 
    THEN ma 
    ELSE ceil(tpl_data_size)::int%ma 
    END
        ) AS tpl_size, 
    bs - page_hdr AS size_per_block, 
    (heappages + toastpages) AS tblpages, 
    heappages,
        toastpages, 
    reltuples, 
    toasttuples, 
    bs, 
    page_hdr, 
    tblid, 
    s.schemaname, 
    s.tblname, 
    s.fillfactor, 
    s.is_na
FROM s
), s3 AS (
SELECT 
    ceil( reltuples / ( (bs-page_hdr)/tpl_size ) ) + ceil( toasttuples / 4 ) AS est_tblpages,
        ceil( reltuples / ( (bs-page_hdr)*s2.fillfactor/(tpl_size*100) ) ) + ceil( toasttuples / 4 ) AS est_tblpages_ff,
        s2.tblpages,
        s2.fillfactor,
        s2.bs,
        s2.tblid,
        s2.schemaname,
        s2.tblname,
        s2.heappages,
        s2.toastpages,
        s2.is_na
FROM s2
) SELECT
    current_database(),
    s3.schemaname, 
    s3.tblname, 
    pg_size_pretty(bs*s3.tblpages) AS real_size,
    pg_size_pretty(((s3.tblpages-est_tblpages)*bs)::bigint) AS extra_size,
    CASE WHEN tblpages - est_tblpages > 0
        THEN 100 * (s3.tblpages - est_tblpages)/s3.tblpages::float
        ELSE 0
    END AS extra_ratio, 
    s3.fillfactor, 
    pg_size_pretty(((s3.tblpages-est_tblpages_ff)*bs)::bigint) AS bloat_size,
    ((tblpages-est_tblpages_ff)*bs)::bigint bytes_bloat_size,
    CASE WHEN s3.tblpages - est_tblpages_ff > 0
        THEN 100 * (s3.tblpages - est_tblpages_ff)/s3.tblpages::float
        ELSE 0
  END AS bloat_ratio, s3.is_na
  FROM s3;

END;
$$ LANGUAGE PLPGSQL;
