CREATE OR REPLACE FUNCTION code.restore_contacts (
    i_user_id bigint,
    i_user_type contacts.user_type,
    i_revision bigint,
    i_x_request_id text
) RETURNS bigint AS $$
DECLARE
    new_revision bigint := impl.acquire_current_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
    change contacts.change_log%ROWTYPE;
    restored boolean := false;
    min_revision bigint;
    max_revision bigint;
    current_earlier_revision bigint;
    current_change_id bigint;
BEGIN
    SELECT min(revision), max(revision)
      FROM contacts.change_log
     WHERE user_id = i_user_id
       AND user_type = i_user_type
      INTO min_revision, max_revision;

    IF NOT EXISTS (
        SELECT 1
          FROM contacts.change_log
         WHERE user_id = i_user_id AND user_type = i_user_type AND revision = i_revision
    ) THEN
        IF revision < min_revision THEN
            RAISE EXCEPTION 'Revision % not found for user_id: %, user_type: %',
                    i_revision, i_user_id, i_user_type
                USING HINT = 'Revision is too old';
        ELSIF revision > max_revision THEN
            RAISE EXCEPTION 'Revision % not found for user_id: %, user_type: %',
                    i_revision, i_user_id, i_user_type
                USING HINT = 'Revision is never existed';
        ELSE
            RAISE EXCEPTION 'Revision % not found for user_id: %, user_type: %',
                    i_revision, i_user_id, i_user_type
                USING HINT = 'Revision should present but it does not';
        END IF;
    END IF;

    SELECT max(change_id)
      FROM contacts.change_log
     WHERE user_id = i_user_id
       AND user_type = i_user_type
       AND revision > i_revision
      INTO current_change_id;

    IF current_change_id IS NULL THEN
        RETURN new_revision - 1;
    END IF;

    LOOP
        SELECT *
          FROM contacts.change_log
         WHERE change_id = current_change_id
          INTO change;

        current_earlier_revision := impl.find_min_earliest_equivalent_contacts_revision (
            i_user_id := i_user_id,
            i_user_type := i_user_type,
            i_current_revision := change.revision,
            i_min_revision := i_revision
        );

        EXIT WHEN current_earlier_revision = i_revision;

        IF current_earlier_revision IS NOT NULL THEN
            SELECT max(change_id)
              FROM contacts.change_log
             WHERE user_id = i_user_id
               AND user_type = i_user_type
               AND revision > i_revision
               AND revision <= current_earlier_revision
              INTO current_change_id;

            CONTINUE;
        END IF;

        IF change.type = 'create_user'::contacts.change_type THEN
            RAISE EXCEPTION 'Can''t revert create_user';
        ELSEIF change.type = 'delete_user'::contacts.change_type THEN
            RAISE EXCEPTION 'Can''t revert delete_user';
        ELSEIF change.type = 'copy_abook'::contacts.change_type THEN
            RAISE EXCEPTION 'Can''t revert copy_abook';
        ELSEIF change.type = 'create_contacts'::contacts.change_type THEN
            PERFORM impl.revert_create_contacts(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'delete_contacts'::contacts.change_type THEN
            PERFORM impl.revert_delete_contacts(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'update_contacts'::contacts.change_type THEN
            PERFORM impl.revert_update_contacts(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'create_list'::contacts.change_type THEN
            PERFORM impl.revert_create_contacts_list(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'delete_list'::contacts.change_type THEN
            PERFORM impl.revert_delete_contacts_list(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'update_list'::contacts.change_type THEN
            PERFORM impl.revert_update_contacts_list(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'create_tag'::contacts.change_type THEN
            PERFORM impl.revert_create_contacts_tag(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'delete_tag'::contacts.change_type THEN
            PERFORM impl.revert_delete_contacts_tag(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'update_tag'::contacts.change_type THEN
            PERFORM impl.revert_update_contacts_tag(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'tag_contacts'::contacts.change_type THEN
            PERFORM impl.revert_tag_contacts(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'untag_contacts'::contacts.change_type THEN
            PERFORM impl.revert_untag_contacts(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'share_list'::contacts.change_type THEN
            PERFORM impl.revert_share_contacts_list(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'revoke_list'::contacts.change_type THEN
            PERFORM impl.revert_revoke_contacts_list(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'subscribe_to_list'::contacts.change_type THEN
            PERFORM impl.revert_subscribe_to_contacts_list(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'revoke_subscribed_list'::contacts.change_type THEN
            PERFORM impl.revert_revoke_subscribed_contacts_list(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'create_emails'::contacts.change_type THEN
            PERFORM impl.revert_create_contacts_emails(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'delete_emails'::contacts.change_type THEN
            PERFORM impl.revert_delete_contacts_emails(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'update_emails'::contacts.change_type THEN
            PERFORM impl.revert_update_contacts_emails(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'tag_emails'::contacts.change_type THEN
            PERFORM impl.revert_tag_contacts_emails(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSEIF change.type = 'untag_emails'::contacts.change_type THEN
            PERFORM impl.revert_untag_contacts_emails(
                i_user_id := i_user_id,
                i_user_type := i_user_type,
                i_revision := new_revision,
                i_x_request_id := i_x_request_id,
                i_change := change
            );
        ELSE
            RAISE EXCEPTION 'Can''t revert %', change.type;
        END IF;

        restored := true;

        SELECT max(change_id)
          FROM contacts.change_log
         WHERE user_id = i_user_id
           AND user_type = i_user_type
           AND revision > i_revision
           AND change_id < current_change_id
          INTO current_change_id;

        EXIT WHEN current_change_id IS NULL;
    END LOOP;

    IF restored THEN
        -- TODO: replace by ON CONFLICT DO NOTHING since PostgreSQL 11
        IF NOT EXISTS (
            SELECT 1
              FROM contacts.equivalent_revisions
             WHERE user_id = i_user_id
               AND user_type = i_user_type
               AND earlier_revision = i_revision
               AND later_revision = new_revision
        ) THEN
            INSERT INTO contacts.equivalent_revisions
                       (  user_id,   user_type, earlier_revision, later_revision)
                VALUES (i_user_id, i_user_type,       i_revision,   new_revision);
        END IF;

        RETURN impl.increment_contacts_revision(
            i_user_id := i_user_id,
            i_user_type := i_user_type
        );
    ELSE
        RETURN new_revision - 1;
    END IF;
END
$$ LANGUAGE plpgsql;
