CREATE OR REPLACE FUNCTION code.update_contacts (
    i_user_id bigint,
    i_user_type contacts.user_type,
    i_contacts code.updated_contact[],
    i_x_request_id text,
    i_revision bigint DEFAULT NULL
) RETURNS bigint AS $$
DECLARE
    new_revision bigint := impl.acquire_current_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
    new_contact_id bigint;
    old_contacts impl.contact_to_update[];
    new_contacts impl.contact[];
    old_contact impl.contact_to_update;
BEGIN
    FOR new_contact_id IN (SELECT DISTINCT contact_id FROM unnest(i_contacts)) LOOP
        PERFORM impl.check_user_contact(
            i_user_id := i_user_id,
            i_user_type := i_user_type,
            i_contact_id := new_contact_id
        );
    END LOOP;

    SELECT array_agg((contact_id, list_id, format, vcard, revision, uri)::impl.contact_to_update)
      FROM contacts.contacts
     WHERE user_id = i_user_id AND user_type = i_user_type AND contact_id = ANY(SELECT contact_id FROM unnest(i_contacts))
      INTO old_contacts;

    IF old_contacts IS NULL OR cardinality(old_contacts) = 0 THEN
        RETURN new_revision - 1;
    END IF;

    IF i_revision IS NOT NULL THEN
        FOREACH old_contact IN ARRAY old_contacts LOOP
            IF old_contact.revision > i_revision THEN
                RAISE EXCEPTION 'Update contact with outdated revision %, when current is %', i_revision, old_contact.revision
                      USING HINT = 'Use revision not less than current for this contact';
            END IF;
        END LOOP;
    END IF;

    new_contacts := (
        SELECT array_agg((
                u.contact_id,
                COALESCE(u.list_id, c.list_id),
                COALESCE(u.format, c.format),
                COALESCE(u.vcard, c.vcard),
                COALESCE(u.uri, c.uri)
            )::impl.contact)
          FROM unnest(i_contacts) AS u
          JOIN contacts.contacts AS c
          ON (c.user_id = i_user_id AND c.user_type = i_user_type AND c.contact_id = u.contact_id)
    );

    IF new_contacts = (
        SELECT array_agg((contact_id, list_id, format, vcard, uri)::impl.contact)
          FROM unnest(old_contacts)
    ) THEN
        RETURN (SELECT max(revision) FROM unnest(old_contacts));
    END IF;

    PERFORM impl.update_contacts(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_revision := new_revision,
        i_contacts := new_contacts
    );

    PERFORM impl.log_contacts_change(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_revision := new_revision,
        i_type := 'update_contacts'::contacts.change_type,
        i_arguments := jsonb_build_object('contacts', i_contacts),
        i_changed := jsonb_build_object('contacts', (
            SELECT array_agg((contact_id, list_id, format, vcard, uri)::code.updated_contact)
              FROM unnest(old_contacts)
        )),
        i_x_request_id := i_x_request_id
    );

    RETURN impl.increment_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
END
$$ LANGUAGE plpgsql;
