CREATE OR REPLACE FUNCTION code.update_contacts_emails (
    i_user_id bigint,
    i_user_type contacts.user_type,
    i_emails code.updated_contacts_email[],
    i_x_request_id text,
    i_revision bigint DEFAULT NULL
) RETURNS bigint AS $$
DECLARE
    new_revision bigint := impl.acquire_current_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
    new_email_id bigint;
    old_emails impl.email_to_update[];
    new_emails impl.email[];
    old_email impl.email_to_update;
BEGIN
    FOR new_email_id IN (SELECT DISTINCT email_id FROM unnest(i_emails)) LOOP
        PERFORM impl.check_user_contacts_email(
            i_user_id := i_user_id,
            i_user_type := i_user_type,
            i_email_id := new_email_id
        );
    END LOOP;

    SELECT array_agg((email_id, contact_id, email, type, label, revision)::impl.email_to_update)
      FROM contacts.emails
     WHERE user_id = i_user_id AND user_type = i_user_type AND email_id = ANY(SELECT email_id FROM unnest(i_emails))
      INTO old_emails;

    IF old_emails IS NULL OR cardinality(old_emails) = 0 THEN
        RETURN new_revision - 1;
    END IF;

    IF i_revision IS NOT NULL THEN
        FOREACH old_email IN ARRAY old_emails LOOP
            IF old_email.revision > i_revision THEN
                RAISE EXCEPTION 'Update email with outdated revision %, when current is %', i_revision, old_email.revision
                      USING HINT = 'Use revision not less than current for this email';
            END IF;
        END LOOP;
    END IF;

    new_emails := (
        SELECT array_agg((
                u.email_id,
                COALESCE(u.contact_id, c.contact_id),
                COALESCE(u.email, c.email),
                COALESCE(u.type, c.type),
                COALESCE(u.label, c.label)
            )::impl.email)
          FROM unnest(i_emails) AS u
          JOIN contacts.emails AS c
          ON (c.user_id = i_user_id AND c.user_type = i_user_type AND c.email_id = u.email_id)
    );

    IF new_emails = (
        SELECT array_agg((email_id, contact_id, email, type, label)::impl.email)
          FROM unnest(old_emails)
    ) THEN
        RETURN (SELECT max(revision) FROM unnest(old_emails));
    END IF;

    PERFORM impl.update_contacts_emails(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_revision := new_revision,
        i_emails := new_emails
    );

    PERFORM impl.log_contacts_change(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_revision := new_revision,
        i_type := 'update_emails'::contacts.change_type,
        i_arguments := jsonb_build_object('emails', i_emails),
        i_changed := jsonb_build_object('emails', (
            SELECT array_agg((email_id, contact_id, email, type, label)::code.updated_contacts_email)
              FROM unnest(old_emails)
        )),
        i_x_request_id := i_x_request_id
    );

    RETURN impl.increment_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
END
$$ LANGUAGE plpgsql;
