CREATE OR REPLACE FUNCTION code.update_contacts_list (
    i_user_id bigint,
    i_user_type contacts.user_type,
    i_list_id bigint,
    i_name text,
    i_x_request_id text,
    i_revision bigint DEFAULT NULL
) RETURNS bigint AS $$
DECLARE
    new_revision bigint := impl.acquire_current_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
    old_name text;
    list_type contacts.list_type;
    old_revision bigint;
BEGIN
    PERFORM impl.check_user_contacts_list(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_list_id := i_list_id
    );

    SELECT name, type, revision
      FROM contacts.lists
     WHERE user_id = i_user_id AND user_type = i_user_type AND list_id = i_list_id
      INTO old_name, list_type, old_revision;

    IF NOT FOUND THEN
        RETURN new_revision - 1;
    END IF;

    IF EXISTS (
        SELECT 1
          FROM code.default_contacts_lists()
         WHERE name = old_name AND type = list_type
    ) THEN
        RAISE EXCEPTION 'Update default list is forbidden'
              USING HINT = 'Update only not default lists';
    END IF;

    IF i_revision IS NOT NULL AND old_revision > i_revision THEN
        RAISE EXCEPTION 'Update list with outdated revision %, when current is %', i_revision, old_revision
              USING HINT = 'Use revision not less than current for this list';
    END IF;

    IF old_name = i_name THEN
        RETURN old_revision;
    END IF;

    UPDATE contacts.lists
       SET name = i_name, revision = new_revision
     WHERE user_id = i_user_id AND user_type = i_user_type AND list_id = i_list_id;

    PERFORM impl.log_contacts_change(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_revision := new_revision,
        i_type := 'update_list'::contacts.change_type,
        i_arguments := jsonb_build_object('list_id', i_list_id, 'name', i_name),
        i_changed := jsonb_build_object('name', old_name),
        i_x_request_id := i_x_request_id
    );

    RETURN impl.increment_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
END
$$ LANGUAGE plpgsql;
