CREATE OR REPLACE FUNCTION code.update_contacts_tag (
    i_user_id bigint,
    i_user_type contacts.user_type,
    i_tag_id bigint,
    i_name text,
    i_x_request_id text,
    i_revision bigint DEFAULT NULL
) RETURNS bigint AS $$
DECLARE
    new_revision bigint := impl.acquire_current_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
    old_name text;
    old_revision bigint;
    tag_name text;
    tag_type contacts.tag_type;
BEGIN
    PERFORM impl.check_user_contacts_tag(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_tag_id := i_tag_id
    );

    SELECT name, revision, name, type
      FROM contacts.tags
     WHERE user_id = i_user_id AND user_type = i_user_type AND tag_id = i_tag_id
      INTO old_name, old_revision, tag_name, tag_type;

    IF NOT FOUND THEN
        RETURN new_revision- 1;
    END IF;

    IF EXISTS (
        SELECT 1
          FROM code.default_contacts_tags()
         WHERE name = tag_name AND type = tag_type
    ) THEN
        RAISE EXCEPTION 'Update default tag is forbidden'
              USING HINT = 'Update only not default tags';
    END IF;

    IF i_revision IS NOT NULL AND old_revision > i_revision THEN
        RAISE EXCEPTION 'Update tag with outdated revision %, when current is %', i_revision, old_revision
              USING HINT = 'Use revision not less than current for this tag';
    END IF;

    IF old_name = i_name THEN
        RETURN old_revision;
    END IF;

    UPDATE contacts.tags
       SET name = i_name, revision = new_revision
     WHERE user_id = i_user_id AND user_type = i_user_type AND tag_id = i_tag_id;

    PERFORM impl.log_contacts_change(
        i_user_id := i_user_id,
        i_user_type := i_user_type,
        i_revision := new_revision,
        i_type := 'update_tag'::contacts.change_type,
        i_arguments := jsonb_build_object('tag_id', i_tag_id, 'name', i_name),
        i_changed := jsonb_build_object('name', old_name),
        i_x_request_id := i_x_request_id
    );

    RETURN impl.increment_contacts_revision(
        i_user_id := i_user_id,
        i_user_type := i_user_type
    );
END
$$ LANGUAGE plpgsql;
