CREATE OR REPLACE FUNCTION code.check_counters(i_uid bigint)
    RETURNS SETOF code.check_record AS $$
DECLARE
    actual_serials mail.serials%rowtype;
BEGIN
    SELECT *
      INTO actual_serials
      FROM mail.serials
     WHERE uid = i_uid;

    IF NOT found THEN
        RAISE EXCEPTION 'Can''t get mail.serials for uid: %', i_uid
             USING HINT = 'Probably user does not exist';
    END IF;

    RETURN QUERY
        SELECT 'mid2serial(mail.box.mid) vs serial.next_mid_serial'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT mx.*, actual_serials.next_mid_serial
              FROM (
                SELECT impl.mid2serial(mid) mid_serial, mid
                  FROM mail.box
                 WHERE uid = i_uid) mx
             WHERE mid_serial >= actual_serials.next_mid_serial) x;

    RETURN QUERY
        SELECT 'folders.{fid, revision} vs serials.{next_fid, next_revision}'::text AS name,
               row_to_json(x.*, true) AS broken
          FROM (
            SELECT fid, actual_serials.next_fid,
                   revision, actual_serials.next_revision
              FROM mail.folders
             WHERE uid = i_uid
               AND (
                revision >= actual_serials.next_revision
                OR
                fid >= actual_serials.next_fid)
            ) x;

    RETURN QUERY
        SELECT 'labels.{lid, revision} vs serials.{next_fid, next_revision}'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT lid, actual_serials.next_lid,
                   revision, actual_serials.next_revision
              FROM mail.labels
             WHERE uid = i_uid
               AND (
                  revision >= actual_serials.next_revision
                  OR
                  lid >= actual_serials.next_lid)
            ) x;

    RETURN QUERY
        SELECT 'tabs.{revision} vs serials.{next_revision}'::text AS name,
               row_to_json(x.*, true) AS broken
          FROM (
            SELECT revision, actual_serials.next_revision
              FROM mail.tabs
             WHERE uid = i_uid
               AND revision >= actual_serials.next_revision
            ) x;

    RETURN QUERY
        SELECT 'counters.{revision} vs serials.{next_revision}'::text AS name,
               row_to_json(x.*, true) AS broken
          FROM (
            SELECT revision, actual_serials.next_revision
              FROM mail.counters
             WHERE uid = i_uid
               AND revision >= actual_serials.next_revision
            ) x;


    RETURN QUERY
        SELECT 'folders'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT f.fid,
                   f.message_count, r_message_count,
                   f.message_seen, r_message_seen,
                   f.message_recent, r_message_recent,
                   f.message_size, r_message_size,
                   f.revision, r_revision,
                   f.next_imap_id, max_imap_id
              FROM mail.folders f
              LEFT JOIN (
                SELECT fid,
                       count(*) as r_message_count,
                       sum(seen::integer) as r_message_seen,
                       sum(size) as r_message_size,
                       sum(recent::integer) as r_message_recent,
                       max(imap_id) as max_imap_id,
                       max(revision) as r_revision
                  FROM mail.box mb
                  JOIN mail.messages mm
                    ON (mb.uid = mm.uid AND mb.mid=mm.mid)
                 WHERE mb.uid = i_uid
                 GROUP BY fid) r
                ON r.fid = f.fid
             WHERE f.uid = i_uid) x
        WHERE
            (
                r_message_count IS NULL
                AND NOT (
                    message_count = 0
                AND message_seen = 0
                AND message_recent = 0
                AND message_size = 0
                AND next_imap_id > 0
                AND revision > 0)
            )
            OR
            (
                r_message_count IS NOT NULL
                AND NOT (
                    message_count = r_message_count
                AND message_recent = r_message_recent
                AND message_seen = r_message_seen
                AND message_size = r_message_size
                AND revision >= r_revision
                AND next_imap_id > max_imap_id)
            );

    RETURN QUERY
        SELECT 'folders.first_unseen'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT f.fid,
                   f.first_unseen, (r).first_unseen AS r_first_unseen,
                   f.first_unseen_id, (r).first_unseen_id AS r_first_unseen_id
              FROM mail.folders f,
                   impl.find_first_unseen(f.uid, f.fid) r
             WHERE uid = i_uid) x
         WHERE first_unseen != r_first_unseen
            OR COALESCE(first_unseen_id, -1) != COALESCE(r_first_unseen_id, -1);

    RETURN QUERY
        SELECT 'folders.missed_system'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT df.name, df.type
              FROM code.default_folders() df
            EXCEPT
            SELECT f.name, f.type
              FROM mail.folders f
             WHERE uid = i_uid) x;


    RETURN QUERY
        SELECT 'labels'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT ld.lid
                   , ld.message_count, r_message_count
                   , ld.message_seen, r_message_seen
              FROM mail.labels ld
              LEFT JOIN (
                SELECT uid, lid, count(*) as r_message_count, sum(seen::integer) as r_message_seen
                  FROM mail.box mb, unnest(lids) AS lid
                 WHERE uid = i_uid
                   AND tid IS NOT NULL
                 GROUP BY uid, lid) m
             USING (uid, lid)
             WHERE ld.uid = i_uid) x
         WHERE message_count != COALESCE(r_message_count, 0)
            OR (message_seen IS NOT NULL AND message_seen != COALESCE(r_message_seen, 0));

    RETURN QUERY
        SELECT 'labels.missed_system'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT dl.name, dl.type
              FROM code.default_labels() dl
            EXCEPT
            SELECT l.name, l.type
              FROM mail.labels l
             WHERE uid = i_uid) x;

    RETURN QUERY
        SELECT 'tabs'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT t.tab,
                   t.message_count, r_message_count,
                   t.message_seen, r_message_seen,
                   t.message_size, r_message_size,
                   t.revision, r_revision
              FROM mail.tabs t
              LEFT JOIN (
                SELECT tab,
                       count(*) as r_message_count,
                       sum(seen::integer) as r_message_seen,
                       sum(size) as r_message_size,
                       max(revision) as r_revision
                  FROM mail.box mb
                  JOIN mail.messages mm
                    ON (mb.uid = mm.uid AND mb.mid=mm.mid)
                 WHERE mb.uid = i_uid
                   AND mb.tab IS NOT NULL
                 GROUP BY tab) r
                ON r.tab = t.tab
             WHERE t.uid = i_uid) x
        WHERE
            (
                r_message_count IS NULL
                AND NOT (
                    message_count = 0
                AND message_seen = 0
                AND message_size = 0
                AND revision > 0)
            )
            OR
            (
                r_message_count IS NOT NULL
                AND NOT (
                    message_count = r_message_count
                AND message_seen = r_message_seen
                AND message_size = r_message_size
                AND revision >= r_revision)
            );

    RETURN QUERY
        SELECT 'box.duplicate_lids'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT mid, lids
              FROM mail.box
             WHERE uid = i_uid
               AND lids IS NOT NULL
               AND #lids != #uniq(sort(lids))) x;

    -- [PS-1544] missed lids
    RETURN QUERY
        SELECT 'box.missed_lids'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT mid, lids, lids - all_lids AS missed_lids
              FROM (
                SELECT array_agg(lid) AS all_lids
                  FROM mail.labels
                 WHERE uid = i_uid) l,
                   mail.box m
             WHERE uid = i_uid
               AND NOT lids <@ all_lids) x;

    RETURN QUERY
        SELECT * FROM code.check_chains(i_uid);

    RETURN QUERY
        SELECT 'threads.labels'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT m.tid,
                   m.labels box_labels,
                   t.labels thread_labels
              FROM (
                SELECT tid,
                       impl.sort_thread_labels(
                            impl.sum(
                                impl.make_thread_labels(lids,1))) AS labels
                  FROM mail.box
                 WHERE uid = i_uid
                   AND tid IS NOT NULL
                   AND #lids > 0
                 GROUP BY tid) m
              FULL JOIN (
                SELECT tid, impl.sort_thread_labels(labels) AS labels
                  FROM mail.threads t
                 WHERE uid = i_uid
                   AND labels IS NOT NULL
                   AND cardinality(labels) > 0) t
                ON (t.tid = m.tid)
             WHERE NOT (t.labels = m.labels)
            ) x;

    RETURN QUERY
        SELECT 'threads'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT t.tid,
                   message_count, r_message_count,
                   message_seen, r_message_seen,
                   newest_mid, r_newest_mid,
                   newest_date, r_newest_date
              FROM mail.threads t, (
                SELECT tid,
                       count(1) r_message_count,
                       sum(seen) r_message_seen,
                       max(newest_date) r_newest_date,
                       max(newest_mid) r_newest_mid
                  FROM (
                    SELECT tid,
                           seen::integer seen,
                           last_value(received_date) over mids_in_thread as newest_date,
                           last_value(mid) over mids_in_thread as newest_mid
                      FROM mail.box
                     WHERE uid = i_uid
                    WINDOW mids_in_thread AS (
                      PARTITION BY tid ORDER BY received_date ROWS
                          BETWEEN UNBOUNDED PRECEDING
                              AND UNBOUNDED FOLLOWING)
                  ) m
                  GROUP by tid) m
             WHERE t.uid = i_uid
               AND m.tid = t.tid
            ) x
            WHERE NOT (
                    message_count = r_message_count
                AND message_seen = r_message_seen
                AND newest_date = r_newest_date
                );

    RETURN QUERY
        SELECT 'box.newest_tif'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT m.fid, m.tid,
                   newest_date, r_newest_date,
                   newest_mid, r_newest_mid
              FROM
                  (SELECT fid, tid,
                          max(newest_date) r_newest_date,
                          max(newest_mid) r_newest_mid
                    FROM (
                      SELECT fid, tid,
                             last_value(received_date) over mids_in_thread as newest_date,
                             last_value(mid) over mids_in_thread as newest_mid
                        FROM mail.box
                       WHERE uid = i_uid
                         AND tid IS NOT NULL
                      WINDOW mids_in_thread AS (
                        PARTITION BY fid, tid ORDER BY received_date ROWS
                            BETWEEN UNBOUNDED PRECEDING
                                AND UNBOUNDED FOLLOWING)) w
                    GROUP by fid, tid) m
              LEFT JOIN
                  (SELECT fid, tid,
                          mid AS newest_mid,
                          received_date AS newest_date
                     FROM mail.box
                    WHERE uid = i_uid
                      AND newest_tif = true) mnt
                ON (m.fid = mnt.fid AND m.tid = mnt.tid)) x
         WHERE (newest_date IS NULL OR r_newest_date != newest_date);

    RETURN QUERY
        SELECT 'box.newest_tit'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT m.tab, m.tid,
                   newest_date, r_newest_date,
                   newest_mid, r_newest_mid
              FROM
                  (SELECT tab, tid,
                          max(newest_date) r_newest_date,
                          max(newest_mid) r_newest_mid
                    FROM (
                      SELECT tab, tid,
                             last_value(received_date) over mids_in_thread as newest_date,
                             last_value(mid) over mids_in_thread as newest_mid
                        FROM mail.box
                       WHERE uid = i_uid
                         AND tab IS NOT NULL
                         AND tid IS NOT NULL
                      WINDOW mids_in_thread AS (
                        PARTITION BY tab, tid ORDER BY received_date ROWS
                            BETWEEN UNBOUNDED PRECEDING
                                AND UNBOUNDED FOLLOWING)) w
                    GROUP by tab, tid) m
              LEFT JOIN
                  (SELECT tab, tid,
                          mid AS newest_mid,
                          received_date AS newest_date
                     FROM mail.box
                    WHERE uid = i_uid
                      AND tab IS NOT NULL
                      AND newest_tit = true) mnt
                ON (m.tab = mnt.tab AND m.tid = mnt.tid)) x
         WHERE (newest_date IS NULL OR r_newest_date != newest_date);

    -- remove it after - MAILPG-1124
    RETURN QUERY
        SELECT 'box.tid_not_present_in_threads'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT DISTINCT tid
              FROM mail.box b
             WHERE uid = i_uid
               AND tid IS NOT NULL
               AND NOT EXISTS (
                SELECT tid FROM mail.threads t
                 WHERE t.tid = b.tid
                   AND t.uid = b.uid)) x;

    RETURN QUERY
        SELECT 'message_in_doomed_folder_without_doom_date'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT mid, doom_date, f.type AS folder_type
              FROM mail.box b
              JOIN mail.folders f
             USING (uid, fid)
             WHERE uid = i_uid
               AND code.is_doom_folder(f.type)
               AND doom_date IS NULL
          ) x;

    RETURN QUERY
        SELECT 'message_in_not_doomed_folder_with_doom_date'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT mid, doom_date, f.type AS folder_type
              FROM mail.box b
              JOIN mail.folders f
             USING (uid, fid)
             WHERE uid = i_uid
               AND NOT code.is_doom_folder(f.type)
               AND doom_date IS NOT NULL
          ) x;

    RETURN QUERY
        SELECT 'message_in_threaded_folder_without_tid'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT mid, doom_date, f.type AS folder_type
              FROM mail.box b
              JOIN mail.folders f
             USING (uid, fid)
             WHERE uid = i_uid
               AND code.is_threaded_folder(f.type)
               AND tid IS NULL
          ) x;

    RETURN QUERY
        SELECT 'message_in_not_threaded_folder_with_tid'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT mid, doom_date, f.type AS folder_type
              FROM mail.box b
              JOIN mail.folders f
             USING (uid, fid)
             WHERE uid = i_uid
               AND NOT code.is_threaded_folder(f.type)
               AND tid IS NOT NULL
          ) x;

    RETURN QUERY
        SELECT 'collectors.collector_id vs serials.next_collector_id'::text AS name,
                row_to_json(x.*, true) AS broken
        FROM (
             SELECT collector_id, actual_serials.next_collector_id
             FROM mail.collectors
             WHERE uid = i_uid
               AND collector_id >= actual_serials.next_collector_id
           ) x;

    RETURN QUERY
        SELECT * FROM code.check_shared_folders_from_owner_view(i_uid);

    RETURN QUERY
        SELECT * FROM code.check_subscribed_folders(i_uid);

    RETURN QUERY
        SELECT 'counters'::text AS name,
                row_to_json(x.*, true) AS broken
          FROM (
            SELECT cc.has_attaches_count, r_message_count,
                   cc.has_attaches_seen, r_message_seen
              FROM (
                SELECT has_attaches_count,
                       has_attaches_seen
                  FROM mail.counters
                 WHERE uid = i_uid
              ) cc CROSS JOIN (
                SELECT COALESCE(count(*), 0) as r_message_count,
                       COALESCE(sum(seen::integer), 0) as r_message_seen
                  FROM mail.box mb
                  JOIN mail.messages mm
                    ON (mb.uid = mm.uid AND mb.mid=mm.mid)
                 WHERE mb.uid = i_uid
                   AND CARDINALITY(mm.attaches) > 0
                   AND NOT impl.is_hidden_folder(impl.get_folder_type(i_uid, mb.fid)) ) r
             ) x
        WHERE
            (
                r_message_count IS NULL
                AND NOT (
                    has_attaches_count = 0
                AND has_attaches_seen = 0)
            )
            OR
            (
                r_message_count IS NOT NULL
                AND NOT (
                    has_attaches_count = r_message_count
                AND has_attaches_seen = r_message_seen)
            );
END;
$$ LANGUAGE plpgsql;