CREATE OR REPLACE FUNCTION code.copy_messages(
    i_uid   code.uid,
    i_mids  code.mids,
    i_dst_fid code.fid,
    i_request_info code.request_info DEFAULT NULL,
    i_dst_tab mail.tab_types DEFAULT NULL
) RETURNS code.operation_result AS $$
DECLARE
    current_revision    bigint;
    i_mid_serial        bigint;
    last_chained        impl.chained_message;
    hide_threads        boolean;
    doom_folder         boolean;
    folder_next_imap_id bigint;
    for_copy            impl.message_for_copy[];
    copies              impl.moved_message[];
BEGIN
    SELECT next_revision,
           next_mid_serial
      INTO current_revision,
           i_mid_serial
      FROM impl.acquire_current_serials(i_uid);

    SELECT NOT code.is_threaded_folder(type), next_imap_id, code.is_doom_folder(type)
      INTO hide_threads, folder_next_imap_id, doom_folder
      FROM mail.folders
     WHERE uid = i_uid
       AND fid = i_dst_fid;

    IF NOT FOUND THEN
        RAISE EXCEPTION 'Nonexistent folder fid: %, uid: %', i_dst_fid, i_uid
              USING HINT = 'Please check your user ID',
                   TABLE = 'mail.folders';
    END IF;

    last_chained := impl.get_last_chained(i_uid, i_dst_fid);

    SELECT array_agg((mid,
                      impl.make_mid(received_date, i_mid_serial + row_num - 1),
                      row_num)::impl.message_for_copy)
      FROM (SELECT mid,
                   row_number() OVER (ORDER BY mid) AS row_num,
                   received_date
              FROM mail.box
             WHERE uid = i_uid
               AND mid = ANY(i_mids)
          ORDER BY mid) t
      INTO for_copy;

    IF for_copy IS NULL OR cardinality(for_copy) = 0 THEN
        RETURN code.unchanged_result();
    END IF;

    INSERT INTO mail.messages (
                uid,
                mid,
                st_id,
                size,
                attributes,
                attaches,
                subject,
                firstline,
                hdr_date,
                hdr_message_id,
                recipients,
                extra_data,
                found_tid,
                thread_rule,
                mime
         )
         SELECT uid,
                fc.dst_mid,
                st_id,
                size,
                array_append(
                    array_remove(attributes, 'copy'::mail.message_attributes),
                    'copy'::mail.message_attributes),
                attaches,
                subject,
                firstline,
                hdr_date,
                hdr_message_id,
                recipients,
                extra_data,
                found_tid,
                thread_rule,
                mime
           FROM mail.messages
           JOIN unnest(for_copy) as fc
             ON src_mid = mid
          WHERE uid = i_uid;

    WITH with_copy AS (
        INSERT INTO mail.box (
                   uid,
                   mid,
                   fid,
                   tid,
                   imap_id,
                   revision,
                   chain,
                   seen,
                   recent,
                   deleted,
                   received_date,
                   lids,
                   doom_date,
                   tab, newest_tit
             )
             SELECT * FROM (
                 SELECT src.uid,
                        fc.dst_mid,
                        i_dst_fid,
                        CASE WHEN hide_threads
                            THEN NULL
                            WHEN src.tid IS NULL
                            THEN impl.get_found_tid(i_uid, src.mid)
                            ELSE src.tid END,
                        /* imap_id = */ folder_next_imap_id + row_num - 1,
                        current_revision,
                        /* chain = */ impl.new_chain_formula(row_num, cardinality(for_copy), last_chained),
                        src.seen,
                        /* recent = */ true,
                        src.deleted,
                        src.received_date,
                        src.lids,
                        CASE WHEN doom_folder
                             THEN now()
                             ELSE NULL END,
                        i_dst_tab, impl.default_newest_tit(i_dst_tab)
                   FROM mail.box as src
                   JOIN unnest(for_copy) as fc
                     ON src_mid = mid
                  WHERE uid = i_uid
              ) t
          RETURNING mid, tid, imap_id, seen, deleted, received_date, lids)
    SELECT array_agg((
        c.mid,
        i_dst_fid,
        NULL, -- source fid
        c.tid,
        NULL,
        c.imap_id,
        NULL,
        NULL,
        c.seen,
        NULL,
        c.deleted,
        c.lids,
        mm.size,
        c.received_date,
        false, -- newest_tif
        impl.get_attaches_info(mm.attaches),
        hdr_message_id,
        mm.attributes,
        i_dst_tab,
        NULL, NULL -- no src_tabs info here
        )::impl.moved_message)
      INTO copies
      FROM with_copy c
      JOIN mail.messages mm
        ON (mm.mid = c.mid AND mm.uid = i_uid);

    IF copies IS NULL OR cardinality(copies) = 0 THEN
        RAISE EXCEPTION 'Messages copied in mail.messages table, but didn''t '
                        'in mail.box';
    END IF;

    INSERT INTO mail.threads_messages
               (uid,        mid, tid, rule, sort_options)
         SELECT uid, fc.dst_mid, tid, rule, sort_options
           FROM mail.threads_messages
           JOIN unnest(for_copy) as fc
             ON src_mid = mid
          WHERE uid = i_uid;

    INSERT INTO mail.message_references
               (uid,        mid, value, type)
         SELECT uid, fc.dst_mid, value, type
           FROM mail.message_references
           JOIN unnest(for_copy) as fc
             ON src_mid = mid
          WHERE uid = i_uid;

    PERFORM impl.update_threads(i_uid, current_revision, copies);

    PERFORM impl.update_dst_folder(
        i_uid,
        i_dst_fid,
        current_revision,
        copies,
        last_chained
    );

    PERFORM impl.update_dst_tab(
        i_uid,
        i_dst_tab,
        current_revision,
        copies
    );

    PERFORM impl.update_labels(i_uid, current_revision, copies);

    PERFORM impl.update_pop3(i_uid, copies);

    PERFORM impl.update_counters(
        i_uid,
        current_revision,
        copies
     );

    PERFORM impl.log_change(
        i_uid          => i_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'copy',
        i_changed      => ARRAY(
            SELECT (mid, dst_tid, i_dst_fid,
                    seen, true, deleted,
                    CASE WHEN dst_tid IS NULL
                         THEN ARRAY[]::integer[]
                         ELSE lids END,
                    hdr_message_id, i_dst_tab)::impl.changed_message
              FROM unnest(copies))::impl.changed_with_fids,
        i_args         => jsonb_build_object(
            'fid', i_dst_fid,
            'mids', i_mids
        )
    );

    UPDATE mail.serials
       SET next_revision = next_revision + 1,
           next_mid_serial = next_mid_serial + cardinality(copies)
     WHERE uid = i_uid;

    RETURN (current_revision, ARRAY(SELECT mid FROM unnest(copies)))::code.operation_result;
END;
$$ LANGUAGE plpgsql;
