CREATE OR REPLACE FUNCTION code.move_messages(
    i_uid   code.uid,
    i_mids  code.mids,
    i_dst_fid code.fid,
    i_request_info code.request_info DEFAULT NULL
) RETURNS code.operation_result AS $$
DECLARE
    current_revision     constant bigint := code.acquire_current_revision(i_uid);
    hide_threads         boolean;
    doom_folder          boolean;
    folder_next_imap_id  bigint;
    last_chained         impl.chained_message;
    moved                impl.moved_message[];
BEGIN
    /*
      1. update mail.box and define what really @moved
      2. update old and new folders
      3. update IMAP chains
      4. fill new threads
      5. update old and new threads_folders
      6. update old threads
      7. labels
      8. revision increment
    */

    SELECT NOT code.is_threaded_folder(type), next_imap_id, code.is_doom_folder(type)
      INTO hide_threads, folder_next_imap_id, doom_folder
      FROM mail.folders
     WHERE uid = i_uid
       AND fid = i_dst_fid;

    IF NOT found THEN
        RAISE EXCEPTION 'Nonexistent folder fid: %, uid: %', i_dst_fid, i_uid
              USING HINT = 'Please check your user ID',
                    TABLE = 'mail.folders';
    END IF;

    last_chained := impl.get_last_chained(i_uid, i_dst_fid);

    WITH with_moved AS (
        UPDATE mail.box m
           SET revision = current_revision,
               fid = i_dst_fid,
               recent = true,
               newest_tif = false,
               tid = CASE WHEN hide_threads
                          THEN NULL
                          WHEN tid IS NULL
                          THEN impl.get_found_tid(i_uid, m.mid)
                          ELSE tid END,
               chain = impl.new_chain_formula(rn, moved_count, last_chained),
               imap_id = (folder_next_imap_id + ml.rn - 1),
               doom_date = CASE WHEN doom_folder
                                THEN now()
                                ELSE NULL END,
               newest_tit = CASE WHEN tab IS NOT NULL
                                 THEN CASE WHEN hide_threads
                                           THEN false
                                           ELSE newest_tit END END
          FROM (
            SELECT mid,
                   row_number() OVER (ORDER BY mid) AS rn,
                   count(1) OVER () AS moved_count,
                   /* get what we change */
                   fid AS src_fid,
                   recent AS old_recent,
                   tid AS src_tid,
                   chain AS src_chain,
                   imap_id AS src_imap_id,
                   newest_tif AS old_newest_tif,
                   tab as src_tab,
                   newest_tit AS old_newest_tit
              FROM mail.box
             WHERE uid = i_uid
               AND fid != i_dst_fid
               AND mid = ANY(i_mids)
          ) ml
         WHERE m.uid = i_uid
           AND m.mid = ml.mid
        RETURNING m.mid, ml.src_fid,
                  m.tid, ml.src_tid,
                  m.imap_id, ml.src_imap_id,
                  ml.src_chain,
                  m.seen, ml.old_recent, m.deleted,
                  m.lids, m.received_date, old_newest_tif,
                  ml.src_tab, old_newest_tit)
    SELECT array_agg((
        um.mid, i_dst_fid, um.src_fid,
        um.tid, um.src_tid,
        um.imap_id, um.src_imap_id,
        um.src_chain,
        um.seen, um.old_recent, deleted,
        um.lids,
        mm.size,
        um.received_date,
        old_newest_tif,
        impl.get_attaches_info(mm.attaches),
        hdr_message_id,
        mm.attributes,
        um.src_tab, um.src_tab, old_newest_tit
        )::impl.moved_message)
      INTO moved
      FROM with_moved um
      JOIN mail.messages mm
        ON (um.mid = mm.mid and mm.uid = i_uid);

    IF moved IS NULL OR cardinality(moved) = 0 THEN
        RETURN code.unchanged_result();
    END IF;


    PERFORM impl.update_src_tabs(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.update_src_folders(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.update_threads(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.update_dst_folder(
        i_uid,
        i_dst_fid,
        current_revision,
        moved,
        last_chained
    );

    PERFORM impl.update_labels(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.update_pop3(
        i_uid,
        moved
    );

    PERFORM impl.update_counters(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.log_change(
        i_uid          => i_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'move',
        i_changed      => ARRAY(
          SELECT (mid, dst_tid, i_dst_fid, src_fid,
                  seen, true, deleted,
                  CASE WHEN dst_tid IS NULL
                       THEN ARRAY[]::integer[]
                       ELSE lids END,
                  dst_tab, src_tab
                 )::impl.changed_message_with_src_fid_tab
           FROM unnest(moved))::impl.changed_with_fids,
        i_args => jsonb_build_object('fid', i_dst_fid)
    );

    PERFORM code.increment_revision(i_uid);

    RETURN (current_revision, ARRAY(SELECT mid FROM unnest(moved)))::code.operation_result;
END;
$$ LANGUAGE plpgsql;
